"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getQueryFilter = exports.findDocuments = void 0;
var _esQuery = require("@kbn/es-query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findDocuments = async ({
  esClient,
  filter,
  page,
  perPage,
  sortField,
  index,
  fields,
  sortOrder,
  logger,
  aggs,
  mSearch
}) => {
  const query = getQueryFilter({
    filter
  });
  let sort;
  const ascOrDesc = sortOrder !== null && sortOrder !== void 0 ? sortOrder : 'asc';
  if (sortField != null) {
    sort = [{
      [sortField]: ascOrDesc
    }];
  } else {
    sort = {
      updated_at: {
        order: 'desc'
      }
    };
  }
  try {
    if (mSearch == null) {
      var _ref, _response$hits$total;
      const response = await esClient.search({
        query,
        track_total_hits: true,
        sort,
        _source: fields !== null && fields !== void 0 && fields.length ? {
          includes: fields
        } : true,
        from: (page - 1) * perPage,
        ignore_unavailable: true,
        index,
        seq_no_primary_term: true,
        size: perPage,
        aggs
      });
      return {
        data: response,
        page,
        perPage,
        total: (_ref = typeof response.hits.total === 'number' ? response.hits.total : (_response$hits$total = response.hits.total) === null || _response$hits$total === void 0 ? void 0 : _response$hits$total.value) !== null && _ref !== void 0 ? _ref : 0
      };
    }
    const mSearchQueryBody = {
      searches: [{
        index
      }, {
        query,
        size: perPage,
        aggs,
        seq_no_primary_term: true,
        from: (page - 1) * perPage,
        sort,
        _source: fields !== null && fields !== void 0 && fields.length ? {
          includes: fields
        } : true
      }, {
        index
      }, {
        query: getQueryFilter({
          filter: mSearch.filter
        }),
        size: mSearch.perPage,
        aggs,
        seq_no_primary_term: true,
        from: (page - 1) * mSearch.perPage,
        sort,
        _source: fields !== null && fields !== void 0 && fields.length ? {
          includes: fields
        } : true
      }],
      ignore_unavailable: true,
      index
    };
    const response = await esClient.msearch(mSearchQueryBody);
    let responseStats = {
      took: 0,
      _shards: {
        total: 0,
        successful: 0,
        skipped: 0,
        failed: 0
      },
      timed_out: false
    };
    // flatten the results of the combined find queries into a single array of hits:
    const results = response.responses.flatMap(res => {
      var _hits$hits;
      const mResponse = res;
      const {
        hits,
        ...responseBody
      } = mResponse;
      // assign whatever the last stats are, they are only used for type
      responseStats = {
        ...responseStats,
        ...responseBody
      };
      return (_hits$hits = hits === null || hits === void 0 ? void 0 : hits.hits) !== null && _hits$hits !== void 0 ? _hits$hits : [];
    });
    return {
      data: {
        ...responseStats,
        hits: {
          hits: results
        }
      },
      page,
      perPage: perPage + mSearch.perPage,
      total: results.length
    };
  } catch (err) {
    logger.error(`Error fetching documents: ${err}`);
    throw err;
  }
};
exports.findDocuments = findDocuments;
const getQueryFilter = ({
  filter
}) => {
  const kqlQuery = filter ? {
    language: 'kuery',
    query: filter
  } : [];
  const config = {
    allowLeadingWildcards: true,
    dateFormatTZ: 'Zulu',
    ignoreFilterIfFieldNotInIndex: false,
    queryStringOptions: {
      analyze_wildcard: true
    }
  };
  return (0, _esQuery.buildEsQuery)(undefined, kqlQuery, [], config);
};
exports.getQueryFilter = getQueryFilter;