"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBenchmarksQuery = exports.getBenchmarksFromAggs = exports.getBenchmarks = void 0;
var _get_grouped_findings_evaluation = require("./get_grouped_findings_evaluation");
var _get_stats = require("./get_stats");
var _get_identifier_runtime_mapping = require("../../../common/runtime_mappings/get_identifier_runtime_mapping");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_BENCHMARKS = 500;
const getBenchmarksQuery = (query, pitId, runtimeMappings) => ({
  size: 0,
  runtime_mappings: {
    ...runtimeMappings,
    ...(0, _get_identifier_runtime_mapping.getIdentifierRuntimeMapping)()
  },
  query,
  aggs: {
    aggs_by_benchmark: {
      terms: {
        field: 'rule.benchmark.id',
        order: {
          _count: 'desc'
        },
        size: MAX_BENCHMARKS
      },
      aggs: {
        aggs_by_benchmark_version: {
          terms: {
            field: 'rule.benchmark.version'
          },
          aggs: {
            aggs_by_benchmark_name: {
              terms: {
                field: 'rule.benchmark.name'
              }
            },
            asset_count: {
              cardinality: {
                field: 'asset_identifier'
              }
            },
            // Result evalution for passed or failed findings
            ..._get_stats.findingsEvaluationAggsQuery,
            // CIS Section Compliance Score and Failed Findings
            ..._get_grouped_findings_evaluation.failedFindingsAggQuery
          }
        }
      }
    }
  },
  pit: {
    id: pitId
  }
});
exports.getBenchmarksQuery = getBenchmarksQuery;
const getBenchmarksFromAggs = benchmarks => {
  return benchmarks.flatMap(benchmarkAggregation => {
    const benchmarkId = benchmarkAggregation.key;
    const versions = benchmarkAggregation.aggs_by_benchmark_version.buckets;
    if (!Array.isArray(versions)) throw new Error('missing aggs by benchmark version');
    return versions.map(version => {
      const benchmarkVersion = version.key;
      const assetCount = version.asset_count.value;
      const resourcesTypesAggs = version.aggs_by_resource_type.buckets;
      let benchmarkName = '';
      if (!Array.isArray(version.aggs_by_benchmark_name.buckets)) throw new Error('missing aggs by benchmark name');
      if (version.aggs_by_benchmark_name && version.aggs_by_benchmark_name.buckets.length > 0) {
        benchmarkName = version.aggs_by_benchmark_name.buckets[0].key;
      }
      if (!Array.isArray(resourcesTypesAggs)) throw new Error('missing aggs by resource type per benchmark');
      const {
        passed_findings: passedFindings,
        failed_findings: failedFindings
      } = version;
      const stats = (0, _get_stats.getStatsFromFindingsEvaluationsAggs)({
        passed_findings: passedFindings,
        failed_findings: failedFindings
      });
      const groupedFindingsEvaluation = (0, _get_grouped_findings_evaluation.getPostureStatsFromAggs)(resourcesTypesAggs);
      return {
        meta: {
          benchmarkId,
          benchmarkVersion,
          benchmarkName,
          assetCount
        },
        stats,
        groupedFindingsEvaluation
      };
    });
  });
};
exports.getBenchmarksFromAggs = getBenchmarksFromAggs;
const getBenchmarks = async (esClient, query, pitId, runtimeMappings, logger) => {
  try {
    var _queryResult$aggregat;
    const queryResult = await esClient.search(getBenchmarksQuery(query, pitId, runtimeMappings));
    const benchmarks = (_queryResult$aggregat = queryResult.aggregations) === null || _queryResult$aggregat === void 0 ? void 0 : _queryResult$aggregat.aggs_by_benchmark.buckets;
    if (!Array.isArray(benchmarks)) throw new Error('missing aggs by benchmark id');
    return getBenchmarksFromAggs(benchmarks);
  } catch (err) {
    logger.error(`Failed to fetch benchmark stats ${err.message}`);
    logger.error(err);
    throw err;
  }
};
exports.getBenchmarks = getBenchmarks;