"use strict";
/**
 * MIT License
 *
 * Copyright (c) 2020-present, Elastic NV
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeMonitorName = exports.isParamOptionSupported = exports.isLightweightMonitorSupported = exports.isBulkAPISupported = exports.generateURL = exports.getSizedBatches = exports.getBatches = exports.printBytes = exports.logGroups = exports.logDiff = void 0;
const semver_1 = __importDefault(require("semver"));
const helpers_1 = require("../helpers");
const colors_1 = require("kleur/colors");
function logDiff(newIDs, changedIDs, removedIDs, unchangedIDs) {
    (0, helpers_1.progress)('Monitor Diff: ' +
        (0, colors_1.green)(`Added(${newIDs.size}) `) +
        (0, colors_1.yellow)(`Updated(${changedIDs.size}) `) +
        (0, colors_1.red)(`Removed(${removedIDs.size}) `) +
        (0, colors_1.grey)(`Unchanged(${unchangedIDs.size})`));
}
exports.logDiff = logDiff;
function logGroups(sizes, newIDs, changedIDs, removedIDs, unchangedIDs) {
    console.groupCollapsed();
    logGroup(sizes, 'Added', newIDs, colors_1.green);
    logGroup(sizes, 'Updated', changedIDs, colors_1.yellow);
    logGroup(sizes, 'Removed', removedIDs, colors_1.red);
    logGroup(sizes, 'Unchanged', unchangedIDs, colors_1.grey);
    console.groupEnd();
}
exports.logGroups = logGroups;
function logGroup(sizes, name, ids, color) {
    if (ids.size === 0)
        return;
    // under collapsed group, so giving 2 space for padding
    printLine(process.stdout.columns - 2);
    console.groupCollapsed(color((0, colors_1.bold)(name)));
    [...ids].forEach(id => {
        console.log((0, colors_1.grey)(`- ${id} (${printBytes(sizes.get(id))})`));
    });
    console.groupEnd();
}
function printLine(length = process.stdout.columns) {
    console.log((0, colors_1.grey)('-').repeat(length));
}
const BYTE_UNITS = ['B', 'kB', 'MB', 'GB', 'TB', 'PB'];
function printBytes(bytes) {
    if (bytes <= 0)
        return '0 B';
    const exponent = Math.min(Math.floor(Math.log10(bytes) / 3), BYTE_UNITS.length - 1);
    bytes /= 1000 ** exponent;
    return `${bytes.toFixed(1)} ${BYTE_UNITS[exponent]}`;
}
exports.printBytes = printBytes;
function getBatches(arr, size) {
    const batches = [];
    for (let i = 0; i < arr.length; i += size) {
        batches.push(arr.slice(i, i + size));
    }
    return batches;
}
exports.getBatches = getBatches;
function getSizedBatches(arr, sizes, maxBatchSizeKB, maxBatchItems) {
    const batches = [];
    let currentBatch = [];
    let currentSize = 0;
    for (const item of arr) {
        const sizeKB = item.id ? Math.round(sizes.get(item.id) / 1000) : 1;
        // If adding the current item would exceed limits, create a new chunk
        if (currentBatch.length >= maxBatchItems ||
            currentSize + sizeKB > maxBatchSizeKB) {
            batches.push(currentBatch);
            currentBatch = [];
            currentSize = 0;
        }
        currentBatch.push(item);
        currentSize += sizeKB;
    }
    // Push the last chunk if it contains any items
    if (currentBatch.length > 0) {
        batches.push(currentBatch);
    }
    return batches;
}
exports.getSizedBatches = getSizedBatches;
function generateURL(options, operation) {
    const url = (0, helpers_1.removeTrailingSlash)(options.url);
    switch (operation) {
        case 'status':
            return `${url}/s/${options.space}/api/stats`;
        case 'bulk_get':
        case 'bulk_update':
        case 'bulk_delete':
            return `${url}/s/${options.space}/api/synthetics/project/${options.id}/monitors`;
        case 'legacy':
            return `${url}/s/${options.space}/api/synthetics/service/project/monitors`;
        case 'location':
            return `${url}/internal/uptime/service/locations`;
        default:
            throw new Error('Invalid operation');
    }
}
exports.generateURL = generateURL;
/**
 * Bulk API is supported for push monitors only from 8.6.0 and above
 */
function isBulkAPISupported(version) {
    return semver_1.default.satisfies(version, '>=8.6.0');
}
exports.isBulkAPISupported = isBulkAPISupported;
/**
 * Lightweight monitors are supported only from 8.5.0 and above
 */
function isLightweightMonitorSupported(monitors, options) {
    const version = options.kibanaVersion;
    return (semver_1.default.satisfies(version, '<8.5.0') &&
        monitors.some(monitor => monitor.type !== 'browser'));
}
exports.isLightweightMonitorSupported = isLightweightMonitorSupported;
/**
 * Lighweight monitor param options are supported only from
 * 8.7.2 and above
 */
function isParamOptionSupported(version) {
    return semver_1.default.satisfies(version, '>=8.7.2');
}
exports.isParamOptionSupported = isParamOptionSupported;
/**
 * Helper that replaces url paths traversal issues when bundling
 */
function normalizeMonitorName(p, replacement = '_') {
    // replace encoded and non encoded dots
    p = p.replace(/%2e|\./gi, replacement);
    // encoded slashes
    p = p.replace(/%2f|%5c/gi, replacement);
    // backslashes
    p = p.replace(/[/\\]+/g, replacement);
    // remove colons
    p = p.replace(/[:]+/g, replacement);
    return p;
}
exports.normalizeMonitorName = normalizeMonitorName;
//# sourceMappingURL=utils.js.map