"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useReindex = void 0;
var _react = require("react");
var _types = require("../../../../../../common/types");
var _types2 = require("../../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POLL_INTERVAL = 3000;
const getReindexState = (reindexState, {
  reindexOp,
  warnings,
  hasRequiredPrivileges,
  meta: updatedMeta
}) => {
  const meta = {
    ...reindexState.meta,
    ...updatedMeta
  };
  // Once we have received an array of existing aliases, we won't update the meta value anymore because
  // when we'll delete the original alias during the reindex process there won't be any aliases pointing
  // to it anymore and the last reindex step (Update existing aliases) would be suddenly removed.
  const aliases = reindexState.meta.aliases.length > 0 ? reindexState.meta.aliases : updatedMeta.aliases;
  const newReindexState = {
    ...reindexState,
    meta: {
      ...meta,
      aliases
    },
    loadingState: _types2.LoadingState.Success
  };
  if (warnings) {
    newReindexState.reindexWarnings = warnings;
  }
  if (hasRequiredPrivileges !== undefined) {
    newReindexState.hasRequiredPrivileges = hasRequiredPrivileges;
  }
  if (reindexOp) {
    // Prevent the UI flickering back to inProgress after cancelling

    let updateLastCompletedStep = true;
    if (reindexOp.lastCompletedStep === _types.ReindexStep.aliasCreated && reindexOp.status !== _types.ReindexStatus.completed) {
      // "ReindexStep.aliasCreated" is the last step coming from the server
      // There is a delay between the moment the server returns that the "lastCompletedStep"
      // is "aliasCreated" and when the server marks reindexing as "completed".
      // We will correct this timing error by only marking the "aliasCreated" step as done
      // when the reindex status is "completed".
      updateLastCompletedStep = false;
    }
    if (updateLastCompletedStep) {
      newReindexState.lastCompletedStep = reindexOp.lastCompletedStep;
    }
    newReindexState.status = reindexOp.status;
    newReindexState.reindexTaskPercComplete = reindexOp.reindexTaskPercComplete;
    newReindexState.errorMessage = reindexOp.errorMessage;

    // if reindex cancellation was "requested" or "loading" and the reindex task is now cancelled,
    // then reindex cancellation has completed, set it to "success"
    if ((reindexState.cancelLoadingState === _types2.CancelLoadingState.Requested || reindexState.cancelLoadingState === _types2.CancelLoadingState.Loading) && reindexOp.status === _types.ReindexStatus.cancelled) {
      newReindexState.cancelLoadingState = _types2.CancelLoadingState.Success;
    } else if (
    // if reindex cancellation has been requested and the reindex task is still in progress,
    // then reindex cancellation has not completed yet, set it to "loading"
    reindexState.cancelLoadingState === _types2.CancelLoadingState.Requested && reindexOp.status === _types.ReindexStatus.inProgress) {
      newReindexState.cancelLoadingState = _types2.CancelLoadingState.Loading;
    } else if (newReindexState.status === _types.ReindexStatus.completed) {
      // The Elasticsearch reindex is complete. We will add one or two (depending if there are
      // existing aliases that need to be updated) "fake" steps only for the UI.
      // This will help our users understand what actually happened in the last step.
      newReindexState.status = _types.ReindexStatus.inProgress;
    }
  }
  return newReindexState;
};
const useReindex = ({
  indexName,
  isFrozen,
  isInDataStream,
  isClosedIndex,
  api
}) => {
  const [reindexState, setReindexState] = (0, _react.useState)({
    loadingState: _types2.LoadingState.Loading,
    errorMessage: null,
    reindexTaskPercComplete: null,
    meta: {
      indexName,
      // these properties will be known after fetching the reindexStatus
      reindexName: '',
      aliases: [],
      isFrozen,
      isInDataStream,
      isClosedIndex,
      isReadonly: false,
      // we don't have this information in the deprecation list
      isFollowerIndex: false // will be updated after fetching the reindexStatus
    }
  });
  const pollIntervalIdRef = (0, _react.useRef)(null);
  const isMounted = (0, _react.useRef)(false);
  const clearPollInterval = (0, _react.useCallback)(() => {
    if (pollIntervalIdRef.current) {
      clearTimeout(pollIntervalIdRef.current);
      pollIntervalIdRef.current = null;
    }
  }, []);

  /**
   * When the server says that reindexing is complete we will fake
   * one (or two in case there are existing aliases to update) extra steps in the UI
   */
  const simulateExtraSteps = (0, _react.useCallback)(() => {
    const delay = 1000;
    const hasExistingAliases = reindexState.meta.aliases.length > 0;

    // Mark "update existing aliases" as completed
    const completeUpdateExistingAliasesStep = () => {
      if (!isMounted.current) {
        return;
      }
      setReindexState(prevValue => {
        return {
          ...prevValue,
          status: _types.ReindexStatus.completed,
          lastCompletedStep: _types.ReindexStep.existingAliasesUpdated
        };
      });
    };

    // Mark "original index deleted" as completed
    const completeDeleteOriginalIndexStep = () => {
      if (!isMounted.current) {
        return;
      }
      setReindexState(prevValue => {
        return {
          ...prevValue,
          status: hasExistingAliases ? _types.ReindexStatus.inProgress : _types.ReindexStatus.completed,
          lastCompletedStep: _types.ReindexStep.originalIndexDeleted
        };
      });
      if (hasExistingAliases) {
        // Still one step to go!
        setTimeout(completeUpdateExistingAliasesStep, delay);
      }
    };
    setTimeout(completeDeleteOriginalIndexStep, delay);
  }, [reindexState.meta.aliases.length]);
  const updateStatus = (0, _react.useCallback)(async () => {
    clearPollInterval();
    const {
      data,
      error
    } = await api.getReindexStatus(indexName);
    if (error) {
      setReindexState(prevValue => {
        return {
          ...prevValue,
          loadingState: _types2.LoadingState.Error,
          errorMessage: error.message.toString(),
          status: _types.ReindexStatus.fetchFailed
        };
      });
      return;
    }
    if (data === null) {
      return;
    }
    setReindexState(prevValue => {
      return getReindexState(prevValue, data);
    });
    if (data.reindexOp && data.reindexOp.status === _types.ReindexStatus.inProgress) {
      // Only keep polling if it exists and is in progress.
      pollIntervalIdRef.current = setTimeout(updateStatus, POLL_INTERVAL);
    } else if (data.reindexOp && data.reindexOp.status === _types.ReindexStatus.completed) {
      simulateExtraSteps();
    }
  }, [clearPollInterval, api, indexName, simulateExtraSteps]);
  const startReindex = (0, _react.useCallback)(async () => {
    setReindexState(prevValue => {
      return {
        ...prevValue,
        // Only reset last completed step if we aren't currently paused
        lastCompletedStep: prevValue.status === _types.ReindexStatus.paused ? prevValue.lastCompletedStep : undefined,
        status: _types.ReindexStatus.inProgress,
        reindexTaskPercComplete: null,
        errorMessage: null,
        cancelLoadingState: undefined
      };
    });
    const {
      data: reindexOp,
      error
    } = await api.startReindexTask(indexName);
    if (error) {
      setReindexState(prevValue => {
        return {
          ...prevValue,
          loadingState: _types2.LoadingState.Error,
          errorMessage: error.message.toString(),
          status: _types.ReindexStatus.failed
        };
      });
      return;
    }
    setReindexState(prevValue => {
      return getReindexState(prevValue, {
        reindexOp,
        meta: prevValue.meta
      });
    });
    updateStatus();
  }, [api, indexName, updateStatus]);
  const cancelReindex = (0, _react.useCallback)(async () => {
    setReindexState(prevValue => {
      return {
        ...prevValue,
        cancelLoadingState: _types2.CancelLoadingState.Requested
      };
    });
    const {
      error
    } = await api.cancelReindexTask(indexName);
    if (error) {
      setReindexState(prevValue => {
        return {
          ...prevValue,
          cancelLoadingState: _types2.CancelLoadingState.Error
        };
      });
      return;
    }
  }, [api, indexName]);
  (0, _react.useEffect)(() => {
    updateStatus();
  }, [updateStatus]);
  (0, _react.useEffect)(() => {
    isMounted.current = true;
    return () => {
      isMounted.current = false;

      // Clean up on unmount.
      clearPollInterval();
    };
  }, [clearPollInterval]);
  return {
    reindexState,
    startReindex,
    cancelReindex,
    updateStatus
  };
};
exports.useReindex = useReindex;