"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usePivotConfig = void 0;
exports.validatePivotConfig = validatePivotConfig;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _fieldTypes = require("@kbn/field-types");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _common = require("../../../../../../../common/types/common");
var _app_dependencies = require("../../../../../app_dependencies");
var _common2 = require("../../../../../common");
var _common3 = require("../common");
var _pivot_aggs = require("../../../../../common/pivot_aggs");
var _pivot_aggs2 = require("../../../../../../../common/types/pivot_aggs");
var _pivot_group_by = require("../../../../../common/pivot_group_by");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Clones aggregation configuration and updates parent references
 * for the sub-aggregations.
 */
function cloneAggItem(item, parentRef) {
  const newItem = {
    ...item
  };
  if (parentRef !== undefined) {
    newItem.parentAgg = parentRef;
  }
  if (newItem.subAggs !== undefined) {
    const newSubAggs = {};
    for (const [key, subItem] of Object.entries(newItem.subAggs)) {
      newSubAggs[key] = cloneAggItem(subItem, newItem);
    }
    newItem.subAggs = newSubAggs;
  }
  return newItem;
}

/**
 * Checks if the aggregations collection is invalid.
 */
function isConfigInvalid(aggsArray) {
  return aggsArray.some(agg => {
    return (0, _pivot_aggs.isPivotAggsWithExtendedForm)(agg) && !agg.isValid() || agg.subAggs && isConfigInvalid(Object.values(agg.subAggs));
  });
}
function validatePivotConfig(config) {
  const valid = Object.values(config.aggregations).length > 0 && Object.values(config.group_by).length > 0;
  const isValid = valid && !isConfigInvalid((0, _common.dictionaryToArray)(config.aggregations));
  return {
    isValid,
    ...(isValid ? {} : {
      errorMessage: _i18n.i18n.translate('xpack.transform.pivotPreview.PivotPreviewIncompleteConfigCalloutBody', {
        defaultMessage: 'Please choose at least one group-by field and aggregation.'
      })
    })
  };
}

/**
 * Returns a root aggregation configuration
 * for provided aggregation item.
 */
function getRootAggregation(item) {
  let rootItem = item;
  while (rootItem.parentAgg !== undefined) {
    rootItem = rootItem.parentAgg;
  }
  return rootItem;
}
const usePivotConfig = (defaults, dataView) => {
  const toastNotifications = (0, _app_dependencies.useToastNotifications)();
  const {
    aggOptions,
    aggOptionsData,
    groupByOptions,
    groupByOptionsData,
    fields
  } = (0, _react.useMemo)(() => (0, _common3.getPivotDropdownOptions)(dataView, defaults.runtimeMappings), [defaults.runtimeMappings, dataView]);

  // The list of selected aggregations
  const [aggList, setAggList] = (0, _react.useState)(defaults.aggList);
  // The list of selected group by fields
  const [groupByList, setGroupByList] = (0, _react.useState)(defaults.groupByList);
  const addGroupBy = (0, _react.useCallback)(d => {
    const label = d[0].label;
    const config = groupByOptionsData[label];
    const aggName = config.aggName;
    const aggNameConflictMessages = (0, _common3.getAggNameConflictToastMessages)(aggName, aggList, groupByList);
    if (aggNameConflictMessages.length > 0) {
      aggNameConflictMessages.forEach(m => toastNotifications.addDanger(m));
      return;
    }
    groupByList[aggName] = config;
    setGroupByList({
      ...groupByList
    });
  }, [aggList, groupByList, groupByOptionsData, toastNotifications]);
  const updateGroupBy = (0, _react.useCallback)((previousAggName, item) => {
    const groupByListWithoutPrevious = {
      ...groupByList
    };
    delete groupByListWithoutPrevious[previousAggName];
    const aggNameConflictMessages = (0, _common3.getAggNameConflictToastMessages)(item.aggName, aggList, groupByListWithoutPrevious);
    if (aggNameConflictMessages.length > 0) {
      aggNameConflictMessages.forEach(m => toastNotifications.addDanger(m));
      return;
    }
    groupByListWithoutPrevious[item.aggName] = item;
    setGroupByList(groupByListWithoutPrevious);
  }, [aggList, groupByList, toastNotifications]);
  const deleteGroupBy = (0, _react.useCallback)(aggName => {
    delete groupByList[aggName];
    setGroupByList({
      ...groupByList
    });
  }, [groupByList]);

  /**
   * Adds an aggregation to the list.
   */
  const addAggregation = (0, _react.useCallback)(d => {
    const label = d[0].label;
    const config = aggOptionsData[label];
    let aggName = config.aggName;
    if ((0, _pivot_aggs.isPivotAggConfigTopMetric)(config)) {
      let suggestedSortField = [...new Set(Object.values(groupByList).map(v => (0, _common2.isPivotGroupByConfigWithUiSupport)(v) ? v.field : undefined))].find(v => {
        var _fields$find;
        return ((_fields$find = fields.find(x => x.name === v)) === null || _fields$find === void 0 ? void 0 : _fields$find.type) === _fieldTypes.KBN_FIELD_TYPES.DATE;
      });
      if (!suggestedSortField) {
        suggestedSortField = [...new Set(Object.values(aggList).map(v => (0, _pivot_group_by.isPivotAggConfigWithUiSupport)(v) ? v.field : undefined).flat().filter(_mlIsDefined.isDefined))].find(v => {
          var _fields$find2;
          return ((_fields$find2 = fields.find(x => x.name === v)) === null || _fields$find2 === void 0 ? void 0 : _fields$find2.type) === _fieldTypes.KBN_FIELD_TYPES.DATE;
        });
      }
      if (suggestedSortField) {
        config.aggConfig.sortField = suggestedSortField;
        config.aggConfig.sortSettings = {};
        config.aggConfig.sortSettings.order = 'desc';
      }
    }
    if (aggList[aggName] && aggName === _pivot_aggs2.PIVOT_SUPPORTED_AGGS.TOP_METRICS) {
      // handle special case for naming top_metric aggs
      const regExp = new RegExp(`^${_pivot_aggs2.PIVOT_SUPPORTED_AGGS.TOP_METRICS}(\\d)*$`);
      const increment = Object.keys(aggList).reduce((acc, curr) => {
        const match = curr.match(regExp);
        if (!match || !match[1]) return acc;
        const n = Number(match[1]);
        return n > acc ? n : acc;
      }, 0);
      aggName = `${_pivot_aggs2.PIVOT_SUPPORTED_AGGS.TOP_METRICS}${increment + 1}`;
      config.aggName = aggName;
    }
    const aggNameConflictMessages = (0, _common3.getAggNameConflictToastMessages)(aggName, aggList, groupByList);
    if (aggNameConflictMessages.length > 0) {
      aggNameConflictMessages.forEach(m => toastNotifications.addDanger(m));
      return;
    }
    aggList[aggName] = config;
    setAggList({
      ...aggList
    });
  }, [aggList, aggOptionsData, groupByList, toastNotifications, fields]);

  /**
   * Adds updated aggregation to the list
   */
  const updateAggregation = (0, _react.useCallback)((previousAggName, item) => {
    const aggListWithoutPrevious = {
      ...aggList
    };
    delete aggListWithoutPrevious[previousAggName];
    const aggNameConflictMessages = (0, _common3.getAggNameConflictToastMessages)(item.aggName, aggListWithoutPrevious, groupByList);
    if (aggNameConflictMessages.length > 0) {
      aggNameConflictMessages.forEach(m => toastNotifications.addDanger(m));
      return;
    }
    aggListWithoutPrevious[item.aggName] = item;
    setAggList(aggListWithoutPrevious);
  }, [aggList, groupByList, toastNotifications]);

  /**
   * Adds sub-aggregation to the aggregation item
   */
  const addSubAggregation = (0, _react.useCallback)((item, d) => {
    var _item$subAggs;
    if (!item.isSubAggsSupported) {
      throw new Error(`Aggregation "${item.agg}" does not support sub-aggregations`);
    }
    const label = d[0].label;
    const config = aggOptionsData[label];
    item.subAggs = (_item$subAggs = item.subAggs) !== null && _item$subAggs !== void 0 ? _item$subAggs : {};
    const aggNameConflictMessages = (0, _common3.getAggNameConflictToastMessages)(config.aggName, item.subAggs, {});
    if (aggNameConflictMessages.length > 0) {
      aggNameConflictMessages.forEach(m => toastNotifications.addDanger(m));
      return;
    }
    item.subAggs[config.aggName] = config;
    const newRootItem = cloneAggItem(getRootAggregation(item));
    updateAggregation(newRootItem.aggName, newRootItem);
  }, [aggOptionsData, toastNotifications, updateAggregation]);

  /**
   * Updates sub-aggregation of the aggregation item
   */
  const updateSubAggregation = (0, _react.useCallback)((prevSubItemName, subItem) => {
    const parent = subItem.parentAgg;
    if (!parent || !parent.subAggs) {
      throw new Error('No parent aggregation reference found');
    }
    const {
      [prevSubItemName]: deleted,
      ...newSubAgg
    } = parent.subAggs;
    const aggNameConflictMessages = (0, _common3.getAggNameConflictToastMessages)(subItem.aggName, newSubAgg, {});
    if (aggNameConflictMessages.length > 0) {
      aggNameConflictMessages.forEach(m => toastNotifications.addDanger(m));
      return;
    }
    parent.subAggs = {
      ...newSubAgg,
      [subItem.aggName]: subItem
    };
    const newRootItem = cloneAggItem(getRootAggregation(subItem));
    updateAggregation(newRootItem.aggName, newRootItem);
  }, [toastNotifications, updateAggregation]);

  /**
   * Deletes sub-aggregation of the aggregation item
   */
  const deleteSubAggregation = (0, _react.useCallback)((item, subAggName) => {
    if (!item.subAggs || !item.subAggs[subAggName]) {
      throw new Error('Unable to delete a sub-agg');
    }
    delete item.subAggs[subAggName];
    const newRootItem = cloneAggItem(getRootAggregation(item));
    updateAggregation(newRootItem.aggName, newRootItem);
  }, [updateAggregation]);

  /**
   * Deletes aggregation from the list
   */
  const deleteAggregation = (0, _react.useCallback)(aggName => {
    delete aggList[aggName];
    setAggList({
      ...aggList
    });
  }, [aggList]);
  const pivotAggsArr = (0, _react.useMemo)(() => (0, _common.dictionaryToArray)(aggList), [aggList]);
  const pivotGroupByArr = (0, _react.useMemo)(() => (0, _common.dictionaryToArray)(groupByList), [groupByList]);
  const requestPayload = (0, _react.useMemo)(() => (0, _common2.getRequestPayload)(pivotAggsArr, pivotGroupByArr), [pivotAggsArr, pivotGroupByArr]);
  const validationStatus = (0, _react.useMemo)(() => {
    return validatePivotConfig(requestPayload.pivot);
  }, [requestPayload]);
  const actions = (0, _react.useMemo)(() => {
    return {
      addAggregation,
      addGroupBy,
      addSubAggregation,
      updateSubAggregation,
      deleteSubAggregation,
      deleteAggregation,
      deleteGroupBy,
      setAggList,
      setGroupByList,
      updateAggregation,
      updateGroupBy
    };
  }, [addAggregation, addGroupBy, addSubAggregation, deleteAggregation, deleteGroupBy, deleteSubAggregation, updateAggregation, updateGroupBy, updateSubAggregation]);
  return (0, _react.useMemo)(() => {
    return {
      actions,
      state: {
        aggList,
        aggOptions,
        aggOptionsData,
        groupByList,
        groupByOptions,
        groupByOptionsData,
        pivotAggsArr,
        pivotGroupByArr,
        validationStatus,
        requestPayload,
        fields
      }
    };
  }, [actions, aggList, aggOptions, aggOptionsData, groupByList, groupByOptions, groupByOptionsData, pivotAggsArr, pivotGroupByArr, validationStatus, requestPayload, fields]);
};
exports.usePivotConfig = usePivotConfig;