"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.serializeXJson = exports.deserializeJson = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const serializeXJson = (v, defaultVal = '{}') => {
  if (!v) {
    return defaultVal;
  }
  if (typeof v === 'string') {
    return formatXJsonString(v);
  }
  return JSON.stringify(v, null, 2);
};
exports.serializeXJson = serializeXJson;
const deserializeJson = input => {
  try {
    return JSON.parse(input);
  } catch (e) {
    return input;
  }
};

/**
 * Format a XJson string input as parsed JSON. Replaces the invalid characters
 *  with a placeholder, parses the new string in a JSON format with the expected
 * indentantion and then replaces the placeholders with the original values.
 */
exports.deserializeJson = deserializeJson;
const formatXJsonString = input => {
  let placeholder = 'PLACEHOLDER';
  const INVALID_STRING_REGEX = /"""(.*?)"""/gs;
  while (input.includes(placeholder)) {
    placeholder += '_';
  }
  const modifiedInput = input.replace(INVALID_STRING_REGEX, () => `"${placeholder}"`);
  let jsonObject;
  try {
    jsonObject = JSON.parse(modifiedInput);
  } catch (error) {
    return input;
  }
  let formattedJsonString = JSON.stringify(jsonObject, null, 2);
  const invalidStrings = input.match(INVALID_STRING_REGEX);
  if (invalidStrings) {
    invalidStrings.forEach(invalidString => {
      formattedJsonString = formattedJsonString.replace(`"${placeholder}"`, invalidString);
    });
  }
  return formattedJsonString;
};