"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeleteSLO = void 0;
var _constants = require("../../common/constants");
var _retry = require("../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DeleteSLO {
  constructor(repository, transformManager, summaryTransformManager, scopedClusterClient, rulesClient, abortController = new AbortController()) {
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.scopedClusterClient = scopedClusterClient;
    this.rulesClient = rulesClient;
    this.abortController = abortController;
  }
  async execute(sloId, options = {
    skipDataDeletion: false,
    skipRuleDeletion: false
  }) {
    const slo = await this.repository.findById(sloId);

    // First delete the linked resources before deleting the data
    const rollupTransformId = (0, _constants.getSLOTransformId)(slo.id, slo.revision);
    const summaryTransformId = (0, _constants.getSLOSummaryTransformId)(slo.id, slo.revision);
    await Promise.all([this.transformManager.uninstall(rollupTransformId), this.summaryTransformManager.uninstall(summaryTransformId), (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.deletePipeline({
      id: (0, _constants.getWildcardPipelineId)(slo.id, slo.revision)
    }, {
      ignore: [404],
      signal: this.abortController.signal
    })), this.repository.deleteById(slo.id, {
      ignoreNotFound: true
    })]);
    await Promise.all([this.deleteRollupData(slo.id, options.skipDataDeletion), this.deleteSummaryData(slo.id, options.skipDataDeletion), this.deleteAssociatedRules(slo.id, options.skipRuleDeletion)]);
  }
  async deleteRollupData(sloId, skip) {
    if (skip) {
      return;
    }
    await this.scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SLI_DESTINATION_INDEX_PATTERN,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: {
            term: {
              'slo.id': sloId
            }
          }
        }
      }
    }, {
      signal: this.abortController.signal
    });
  }
  async deleteSummaryData(sloId, skip) {
    if (skip) {
      return;
    }
    await this.scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: false,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: {
            term: {
              'slo.id': sloId
            }
          }
        }
      }
    }, {
      signal: this.abortController.signal
    });
  }
  async deleteAssociatedRules(sloId, skip) {
    if (skip) {
      return;
    }
    try {
      await this.rulesClient.bulkDeleteRules({
        filter: `alert.attributes.params.sloId:${sloId}`
      });
    } catch (err) {
      // no-op
    }
  }
}
exports.DeleteSLO = DeleteSLO;