"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerApiKeyRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _error_handler = require("../utils/error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerApiKeyRoutes = ({
  logger,
  router,
  getSecurity
}) => {
  router.get({
    path: '/internal/serverless_search/api_keys',
    validate: {},
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the es client'
      }
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (user) {
      const privileges = await client.asCurrentUser.security.hasPrivileges({
        cluster: ['manage_own_api_key']
      });
      const canManageOwnApiKey = privileges === null || privileges === void 0 ? void 0 : privileges.cluster.manage_own_api_key;
      try {
        const apiKeys = await client.asCurrentUser.security.getApiKey({
          username: user.username
        });
        const validKeys = apiKeys.api_keys.filter(({
          invalidated
        }) => !invalidated);
        return response.ok({
          body: {
            apiKeys: validKeys,
            canManageOwnApiKey
          }
        });
      } catch {
        return response.ok({
          body: {
            apiKeys: [],
            canManageOwnApiKey
          }
        });
      }
    }
    return response.customError({
      statusCode: 502,
      body: 'Could not retrieve current user, security plugin is not ready'
    });
  }));
  router.post({
    path: '/internal/serverless_search/api_key',
    validate: {
      body: _configSchema.schema.any()
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the es client'
      }
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const security = await getSecurity();
    const result = await security.authc.apiKeys.create(request, request.body);
    if (result) {
      const apiKey = {
        ...result,
        beats_logstash_format: `${result.id}:${result.api_key}`
      };
      return response.ok({
        body: apiKey
      });
    }
    return response.customError({
      statusCode: 502,
      body: 'Could not retrieve current user, security plugin is not ready'
    });
  }));
};
exports.registerApiKeyRoutes = registerApiKeyRoutes;