"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.savePinnedEvents = exports.pickSavedPinnedEvent = exports.persistPinnedEventOnTimeline = exports.getAllPinnedEventsByTimelineId = exports.deletePinnedEventOnTimeline = exports.deleteAllPinnedEventsOnTimeline = exports.convertSavedObjectToSavedPinnedEvent = exports.PINNED_EVENTS_PER_PAGE = void 0;
var _PathReporter = require("io-ts/lib/PathReporter");
var _pipeable = require("fp-ts/pipeable");
var _Either = require("fp-ts/Either");
var _function = require("fp-ts/function");
var _constants = require("../../../../../common/constants");
var _saved_object = require("../../../../../common/types/timeline/pinned_event/saved_object");
var _pinned_events = require("../../saved_object_mappings/pinned_events");
var _field_migrator = require("./field_migrator");
var _saved_object_mappings = require("../../saved_object_mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deletePinnedEventOnTimeline = async (request, pinnedEventIds) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  await Promise.all(pinnedEventIds.map(pinnedEventId => savedObjectsClient.delete(_pinned_events.pinnedEventSavedObjectType, pinnedEventId)));
};
exports.deletePinnedEventOnTimeline = deletePinnedEventOnTimeline;
const deleteAllPinnedEventsOnTimeline = async (request, timelineId) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const options = {
    type: _pinned_events.pinnedEventSavedObjectType,
    hasReference: {
      type: _saved_object_mappings.timelineSavedObjectType,
      id: timelineId
    }
  };
  const pinnedEventToBeDeleted = await getAllSavedPinnedEvents(request, options);
  await Promise.all(pinnedEventToBeDeleted.map(pinnedEvent => savedObjectsClient.delete(_pinned_events.pinnedEventSavedObjectType, pinnedEvent.pinnedEventId)));
};
exports.deleteAllPinnedEventsOnTimeline = deleteAllPinnedEventsOnTimeline;
const PINNED_EVENTS_PER_PAGE = exports.PINNED_EVENTS_PER_PAGE = 10000; // overrides the saved object client's FIND_DEFAULT_PER_PAGE (20)

const getAllPinnedEventsByTimelineId = async (request, timelineId) => {
  const options = {
    type: _pinned_events.pinnedEventSavedObjectType,
    hasReference: {
      type: _saved_object_mappings.timelineSavedObjectType,
      id: timelineId
    },
    perPage: PINNED_EVENTS_PER_PAGE
  };
  return getAllSavedPinnedEvents(request, options);
};
exports.getAllPinnedEventsByTimelineId = getAllPinnedEventsByTimelineId;
const persistPinnedEventOnTimeline = async (request, pinnedEventId, eventId, timelineId) => {
  if (pinnedEventId != null) {
    // Delete Pinned Event on Timeline
    await deletePinnedEventOnTimeline(request, [pinnedEventId]);
    return {
      unpinned: true
    };
  }
  const pinnedEvents = await getPinnedEventsInTimelineWithEventId(request, timelineId, eventId);

  // we already had this event pinned so let's just return the one we already had
  if (pinnedEvents.length > 0) {
    return {
      ...pinnedEvents[0]
    };
  }
  return createPinnedEvent({
    request,
    eventId,
    timelineId
  });
};
exports.persistPinnedEventOnTimeline = persistPinnedEventOnTimeline;
const getPinnedEventsInTimelineWithEventId = async (request, timelineId, eventId) => {
  const allPinnedEventId = await getAllPinnedEventsByTimelineId(request, timelineId);
  const pinnedEvents = allPinnedEventId.filter(pinnedEvent => pinnedEvent.eventId === eventId);
  return pinnedEvents;
};
const createPinnedEvent = async ({
  request,
  eventId,
  timelineId
}) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const savedPinnedEvent = {
    eventId,
    timelineId
  };
  const pinnedEventWithCreator = pickSavedPinnedEvent(null, savedPinnedEvent, request.user);
  const {
    transformedFields: migratedAttributes,
    references
  } = _field_migrator.pinnedEventFieldsMigrator.extractFieldsToReferences({
    data: pinnedEventWithCreator
  });
  const pinnedEventAttributes = {
    eventId: migratedAttributes.eventId,
    created: migratedAttributes.created,
    createdBy: migratedAttributes.createdBy,
    updated: migratedAttributes.updated,
    updatedBy: migratedAttributes.updatedBy
  };
  const createdPinnedEvent = await savedObjectsClient.create(_pinned_events.pinnedEventSavedObjectType, pinnedEventAttributes, {
    references
  });
  const repopulatedSavedObject = _field_migrator.pinnedEventFieldsMigrator.populateFieldsFromReferences(createdPinnedEvent);

  // create Pinned Event on Timeline
  return {
    ...convertSavedObjectToSavedPinnedEvent(repopulatedSavedObject)
  };
};
const getAllSavedPinnedEvents = async (request, options) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const savedObjects = await savedObjectsClient.find(options);
  return savedObjects.saved_objects.map(savedObject => {
    const populatedPinnedEvent = _field_migrator.pinnedEventFieldsMigrator.populateFieldsFromReferences(savedObject);
    return convertSavedObjectToSavedPinnedEvent(populatedPinnedEvent);
  });
};
const savePinnedEvents = (frameworkRequest, timelineSavedObjectId, pinnedEventIds) => Promise.all(pinnedEventIds.map(eventId => persistPinnedEventOnTimeline(frameworkRequest, null,
// pinnedEventSavedObjectId
eventId, timelineSavedObjectId)));
exports.savePinnedEvents = savePinnedEvents;
const convertSavedObjectToSavedPinnedEvent = savedObject => (0, _pipeable.pipe)(_saved_object.SavedObjectPinnedEventRuntimeType.decode(savedObject), (0, _Either.map)(savedPinnedEvent => {
  return {
    pinnedEventId: savedPinnedEvent.id,
    version: savedPinnedEvent.version,
    timelineId: savedPinnedEvent.attributes.timelineId,
    created: savedPinnedEvent.attributes.created,
    createdBy: savedPinnedEvent.attributes.createdBy,
    eventId: savedPinnedEvent.attributes.eventId,
    updated: savedPinnedEvent.attributes.updated,
    updatedBy: savedPinnedEvent.attributes.updatedBy
  };
}), (0, _Either.fold)(errors => {
  throw new Error((0, _PathReporter.failure)(errors).join('\n'));
}, _function.identity));
exports.convertSavedObjectToSavedPinnedEvent = convertSavedObjectToSavedPinnedEvent;
const pickSavedPinnedEvent = (pinnedEventId, savedPinnedEvent, userInfo) => {
  var _userInfo$username2;
  const dateNow = new Date().valueOf();
  if (pinnedEventId == null) {
    var _userInfo$username;
    savedPinnedEvent.created = dateNow;
    savedPinnedEvent.createdBy = (_userInfo$username = userInfo === null || userInfo === void 0 ? void 0 : userInfo.username) !== null && _userInfo$username !== void 0 ? _userInfo$username : _constants.UNAUTHENTICATED_USER;
  }
  savedPinnedEvent.updated = dateNow;
  savedPinnedEvent.updatedBy = (_userInfo$username2 = userInfo === null || userInfo === void 0 ? void 0 : userInfo.username) !== null && _userInfo$username2 !== void 0 ? _userInfo$username2 : _constants.UNAUTHENTICATED_USER;
  return savedPinnedEvent;
};
exports.pickSavedPinnedEvent = pickSavedPinnedEvent;