"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getValidationNode = void 0;
var _fp = require("lodash/fp");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This node runs all validation steps, and will redirect to the END of the graph if no errors are found.
 * Any new validation steps should be added here.
 */
const getValidationNode = ({
  logger
}) => {
  return async state => {
    const query = state.elastic_rule.query;

    // We want to prevent infinite loops, so we increment the iterations counter for each validation run.
    const currentIteration = state.validation_errors.iterations + 1;
    let esqlErrors = '';
    try {
      const sanitizedQuery = query ? removePlaceHolders(query) : '';
      if (!(0, _fp.isEmpty)(sanitizedQuery)) {
        const {
          errors,
          isEsqlQueryAggregating,
          hasMetadataOperator
        } = (0, _securitysolutionUtils.parseEsqlQuery)(sanitizedQuery);
        if (!(0, _fp.isEmpty)(errors)) {
          esqlErrors = JSON.stringify(errors);
        } else if (!isEsqlQueryAggregating && !hasMetadataOperator) {
          esqlErrors = `Queries that do't use the STATS...BY function (non-aggregating queries) must include the "metadata _id, _version, _index" operator after the source command. For example: FROM logs* metadata _id, _version, _index.`;
        }
      }
      if (esqlErrors) {
        logger.debug(`ESQL query validation failed: ${esqlErrors}`);
      }
    } catch (error) {
      esqlErrors = error.message;
      logger.info(`Error parsing ESQL query: ${error.message}`);
    }
    return {
      validation_errors: {
        iterations: currentIteration,
        esql_errors: esqlErrors
      }
    };
  };
};
exports.getValidationNode = getValidationNode;
function removePlaceHolders(query) {
  return query.replaceAll(/\[(macro|lookup):.*?\]/g, '') // Removes any macro or lookup placeholders
  .replaceAll(/\n(\s*?\|\s*?\n)*/g, '\n'); // Removes any empty lines with | (pipe) alone after removing the placeholders
}