"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationsDataRulesClient = void 0;
var _constants = require("../../../../../common/siem_migrations/constants");
var _sort = require("./sort");
var _search = require("./search");
var _rule_migrations_data_base_client = require("./rule_migrations_data_base_client");
var _constants2 = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* BULK_MAX_SIZE defines the number to break down the bulk operations by.
 * The 500 number was chosen as a reasonable number to avoid large payloads. It can be adjusted if needed. */
const BULK_MAX_SIZE = 500;
/* DEFAULT_SEARCH_BATCH_SIZE defines the default number of documents to retrieve per search operation
 * when retrieving search results in batches. */
const DEFAULT_SEARCH_BATCH_SIZE = 500;
class RuleMigrationsDataRulesClient extends _rule_migrations_data_base_client.RuleMigrationsDataBaseClient {
  /** Indexes an array of rule migrations to be processed */
  async create(ruleMigrations) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    let ruleMigrationsSlice;
    const createdAt = new Date().toISOString();
    while ((ruleMigrationsSlice = ruleMigrations.splice(0, BULK_MAX_SIZE)).length) {
      await this.esClient.bulk({
        refresh: 'wait_for',
        operations: ruleMigrationsSlice.flatMap(ruleMigration => [{
          create: {
            _index: index
          }
        }, {
          ...ruleMigration,
          '@timestamp': createdAt,
          status: _constants.SiemMigrationStatus.PENDING,
          created_by: profileId,
          updated_by: profileId,
          updated_at: createdAt
        }])
      }).catch(error => {
        this.logger.error(`Error creating rule migrations: ${error.message}`);
        throw error;
      });
    }
  }

  /** Updates an array of rule migrations to be processed */
  async update(ruleMigrations) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    let ruleMigrationsSlice;
    const updatedAt = new Date().toISOString();
    while ((ruleMigrationsSlice = ruleMigrations.splice(0, BULK_MAX_SIZE)).length) {
      await this.esClient.bulk({
        refresh: 'wait_for',
        operations: ruleMigrationsSlice.flatMap(ruleMigration => {
          const {
            id,
            ...rest
          } = ruleMigration;
          return [{
            update: {
              _index: index,
              _id: id
            }
          }, {
            doc: {
              ...rest,
              updated_by: profileId,
              updated_at: updatedAt
            }
          }];
        })
      }).catch(error => {
        this.logger.error(`Error updating rule migrations: ${error.message}`);
        throw error;
      });
    }
  }

  /** Retrieves an array of rule documents of a specific migrations */
  async get(migrationId, {
    filters = {},
    sort: sortParam = {},
    from,
    size
  } = {}) {
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId, filters);
    const sort = sortParam.sortField ? (0, _sort.getSortingOptions)(sortParam) : undefined;
    const result = await this.esClient.search({
      index,
      query,
      sort,
      from,
      size
    }).catch(error => {
      this.logger.error(`Error searching rule migrations: ${error.message}`);
      throw error;
    });
    return {
      total: this.getTotalHits(result),
      data: this.processResponseHits(result)
    };
  }

  /** Returns batching functions to traverse all the migration rules search results */
  searchBatches(migrationId, options = {}) {
    const {
      size = DEFAULT_SEARCH_BATCH_SIZE,
      filters = {},
      scroll
    } = options;
    const query = this.getFilterQuery(migrationId, filters);
    const search = {
      query,
      sort: '_doc',
      scroll,
      size
    }; // sort by _doc to ensure consistent order
    try {
      return this.getSearchBatches(search);
    } catch (error) {
      this.logger.error(`Error scrolling rule migrations: ${error.message}`);
      throw error;
    }
  }

  /** Updates one rule migration status to `processing` */
  async saveProcessing(id) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const doc = {
      status: _constants.SiemMigrationStatus.PROCESSING,
      updated_by: profileId,
      updated_at: new Date().toISOString()
    };
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for'
    }).catch(error => {
      this.logger.error(`Error updating rule migration status to processing: ${error.message}`);
      throw error;
    });
  }

  /** Updates one rule migration with the provided data and sets the status to `completed` */
  async saveCompleted({
    id,
    ...ruleMigration
  }) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const doc = {
      ...ruleMigration,
      status: _constants.SiemMigrationStatus.COMPLETED,
      updated_by: profileId,
      updated_at: new Date().toISOString()
    };
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for'
    }).catch(error => {
      this.logger.error(`Error updating rule migration status to completed: ${error.message}`);
      throw error;
    });
  }

  /** Updates one rule migration with the provided data and sets the status to `failed` */
  async saveError({
    id,
    ...ruleMigration
  }) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const doc = {
      ...ruleMigration,
      status: _constants.SiemMigrationStatus.FAILED,
      updated_by: profileId,
      updated_at: new Date().toISOString()
    };
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for'
    }).catch(error => {
      this.logger.error(`Error updating rule migration status to failed: ${error.message}`);
      throw error;
    });
  }

  /** Updates all the rule migration with the provided id with status `processing` back to `pending` */
  async releaseProcessing(migrationId) {
    return this.updateStatus(migrationId, {
      status: _constants.SiemMigrationStatus.PROCESSING
    }, _constants.SiemMigrationStatus.PENDING);
  }

  /** Updates all the rule migration with the provided id and with status `statusToQuery` to `statusToUpdate` */
  async updateStatus(migrationId, filter, statusToUpdate, {
    refresh = false
  } = {}) {
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId, filter);
    const script = {
      source: `ctx._source['status'] = '${statusToUpdate}'`
    };
    await this.esClient.updateByQuery({
      index,
      query,
      script,
      refresh
    }).catch(error => {
      this.logger.error(`Error updating rule migrations status: ${error.message}`);
      throw error;
    });
  }

  /** Retrieves the translation stats for the rule migrations with the provided id */
  async getTranslationStats(migrationId) {
    var _result$aggregations, _doc_count, _doc_count2, _successAgg$installab, _doc_count3, _successAgg$prebuilt, _doc_count4, _aggs$failed;
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId);
    const aggregations = {
      success: {
        filter: {
          term: {
            status: _constants.SiemMigrationStatus.COMPLETED
          }
        },
        aggs: {
          result: {
            terms: {
              field: 'translation_result'
            }
          },
          installable: {
            filter: {
              bool: {
                must: _search.conditions.isInstallable()
              }
            }
          },
          prebuilt: {
            filter: _search.conditions.isPrebuilt()
          }
        }
      },
      failed: {
        filter: {
          term: {
            status: _constants.SiemMigrationStatus.FAILED
          }
        }
      }
    };
    const result = await this.esClient.search({
      index,
      query,
      aggregations,
      _source: false
    }).catch(error => {
      this.logger.error(`Error getting rule migrations stats: ${error.message}`);
      throw error;
    });
    const aggs = (_result$aggregations = result.aggregations) !== null && _result$aggregations !== void 0 ? _result$aggregations : {};
    const total = this.getTotalHits(result);
    const successAgg = aggs.success;
    const translationResultsAgg = successAgg.result;
    return {
      id: migrationId,
      rules: {
        total,
        success: {
          total: (_doc_count = successAgg === null || successAgg === void 0 ? void 0 : successAgg.doc_count) !== null && _doc_count !== void 0 ? _doc_count : 0,
          result: this.translationResultAggCount(translationResultsAgg),
          installable: (_doc_count2 = (_successAgg$installab = successAgg.installable) === null || _successAgg$installab === void 0 ? void 0 : _successAgg$installab.doc_count) !== null && _doc_count2 !== void 0 ? _doc_count2 : 0,
          prebuilt: (_doc_count3 = (_successAgg$prebuilt = successAgg.prebuilt) === null || _successAgg$prebuilt === void 0 ? void 0 : _successAgg$prebuilt.doc_count) !== null && _doc_count3 !== void 0 ? _doc_count3 : 0
        },
        failed: (_doc_count4 = (_aggs$failed = aggs.failed) === null || _aggs$failed === void 0 ? void 0 : _aggs$failed.doc_count) !== null && _doc_count4 !== void 0 ? _doc_count4 : 0
      }
    };
  }

  /** Retrieves the stats for the rule migrations with the provided id */
  async getStats(migrationId) {
    var _result$aggregations2, _value_as_string, _aggs$createdAt, _value_as_string2, _aggs$lastUpdatedAt;
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId);
    const aggregations = {
      status: {
        terms: {
          field: 'status'
        }
      },
      createdAt: {
        min: {
          field: '@timestamp'
        }
      },
      lastUpdatedAt: {
        max: {
          field: 'updated_at'
        }
      }
    };
    const result = await this.esClient.search({
      index,
      query,
      aggregations,
      _source: false
    }).catch(error => {
      this.logger.error(`Error getting rule migrations stats: ${error.message}`);
      throw error;
    });
    const aggs = (_result$aggregations2 = result.aggregations) !== null && _result$aggregations2 !== void 0 ? _result$aggregations2 : {};
    return {
      id: migrationId,
      rules: {
        total: this.getTotalHits(result),
        ...this.statusAggCounts(aggs.status)
      },
      created_at: (_value_as_string = (_aggs$createdAt = aggs.createdAt) === null || _aggs$createdAt === void 0 ? void 0 : _aggs$createdAt.value_as_string) !== null && _value_as_string !== void 0 ? _value_as_string : '',
      last_updated_at: (_value_as_string2 = (_aggs$lastUpdatedAt = aggs.lastUpdatedAt) === null || _aggs$lastUpdatedAt === void 0 ? void 0 : _aggs$lastUpdatedAt.value_as_string) !== null && _value_as_string2 !== void 0 ? _value_as_string2 : ''
    };
  }

  /** Retrieves the stats for all the rule migrations aggregated by migration id, in creation order */
  async getAllStats() {
    var _result$aggregations3, _ref;
    const index = await this.getIndexName();
    const aggregations = {
      migrationIds: {
        terms: {
          field: 'migration_id',
          order: {
            createdAt: 'asc'
          },
          size: _constants2.MAX_ES_SEARCH_SIZE
        },
        aggregations: {
          status: {
            terms: {
              field: 'status'
            }
          },
          createdAt: {
            min: {
              field: '@timestamp'
            }
          },
          lastUpdatedAt: {
            max: {
              field: 'updated_at'
            }
          }
        }
      }
    };
    const result = await this.esClient.search({
      index,
      aggregations,
      _source: false
    }).catch(error => {
      this.logger.error(`Error getting all rule migrations stats: ${error.message}`);
      throw error;
    });
    const migrationsAgg = (_result$aggregations3 = result.aggregations) === null || _result$aggregations3 === void 0 ? void 0 : _result$aggregations3.migrationIds;
    const buckets = (_ref = migrationsAgg === null || migrationsAgg === void 0 ? void 0 : migrationsAgg.buckets) !== null && _ref !== void 0 ? _ref : [];
    return buckets.map(bucket => {
      var _bucket$createdAt, _bucket$lastUpdatedAt;
      return {
        id: `${bucket.key}`,
        rules: {
          total: bucket.doc_count,
          ...this.statusAggCounts(bucket.status)
        },
        created_at: (_bucket$createdAt = bucket.createdAt) === null || _bucket$createdAt === void 0 ? void 0 : _bucket$createdAt.value_as_string,
        last_updated_at: (_bucket$lastUpdatedAt = bucket.lastUpdatedAt) === null || _bucket$lastUpdatedAt === void 0 ? void 0 : _bucket$lastUpdatedAt.value_as_string
      };
    });
  }

  /** Retrieves the stats for the integrations of all the migration rules */
  async getAllIntegrationsStats() {
    var _result$aggregations4, _ref2;
    const index = await this.getIndexName();
    const aggregations = {
      integrationIds: {
        terms: {
          field: 'elastic_rule.integration_ids',
          // aggregate by integration ids
          exclude: '',
          // excluding empty string integration ids
          size: _constants2.MAX_ES_SEARCH_SIZE
        }
      }
    };
    const result = await this.esClient.search({
      index,
      aggregations,
      _source: false
    }).catch(error => {
      this.logger.error(`Error getting all integrations stats: ${error.message}`);
      throw error;
    });
    const integrationsAgg = (_result$aggregations4 = result.aggregations) === null || _result$aggregations4 === void 0 ? void 0 : _result$aggregations4.integrationIds;
    const buckets = (_ref2 = integrationsAgg === null || integrationsAgg === void 0 ? void 0 : integrationsAgg.buckets) !== null && _ref2 !== void 0 ? _ref2 : [];
    return buckets.map(bucket => ({
      id: `${bucket.key}`,
      total_rules: bucket.doc_count
    }));
  }
  statusAggCounts(statusAgg) {
    var _buckets$find$doc_cou, _buckets$find, _buckets$find$doc_cou2, _buckets$find2, _buckets$find$doc_cou3, _buckets$find3, _buckets$find$doc_cou4, _buckets$find4;
    const buckets = statusAgg.buckets;
    return {
      [_constants.SiemMigrationStatus.PENDING]: (_buckets$find$doc_cou = (_buckets$find = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.PENDING)) === null || _buckets$find === void 0 ? void 0 : _buckets$find.doc_count) !== null && _buckets$find$doc_cou !== void 0 ? _buckets$find$doc_cou : 0,
      [_constants.SiemMigrationStatus.PROCESSING]: (_buckets$find$doc_cou2 = (_buckets$find2 = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.PROCESSING)) === null || _buckets$find2 === void 0 ? void 0 : _buckets$find2.doc_count) !== null && _buckets$find$doc_cou2 !== void 0 ? _buckets$find$doc_cou2 : 0,
      [_constants.SiemMigrationStatus.COMPLETED]: (_buckets$find$doc_cou3 = (_buckets$find3 = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.COMPLETED)) === null || _buckets$find3 === void 0 ? void 0 : _buckets$find3.doc_count) !== null && _buckets$find$doc_cou3 !== void 0 ? _buckets$find$doc_cou3 : 0,
      [_constants.SiemMigrationStatus.FAILED]: (_buckets$find$doc_cou4 = (_buckets$find4 = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.FAILED)) === null || _buckets$find4 === void 0 ? void 0 : _buckets$find4.doc_count) !== null && _buckets$find$doc_cou4 !== void 0 ? _buckets$find$doc_cou4 : 0
    };
  }
  translationResultAggCount(resultAgg) {
    var _buckets$find$doc_cou5, _buckets$find5, _buckets$find$doc_cou6, _buckets$find6, _buckets$find$doc_cou7, _buckets$find7;
    const buckets = resultAgg.buckets;
    return {
      [_constants.RuleTranslationResult.FULL]: (_buckets$find$doc_cou5 = (_buckets$find5 = buckets.find(({
        key
      }) => key === _constants.RuleTranslationResult.FULL)) === null || _buckets$find5 === void 0 ? void 0 : _buckets$find5.doc_count) !== null && _buckets$find$doc_cou5 !== void 0 ? _buckets$find$doc_cou5 : 0,
      [_constants.RuleTranslationResult.PARTIAL]: (_buckets$find$doc_cou6 = (_buckets$find6 = buckets.find(({
        key
      }) => key === _constants.RuleTranslationResult.PARTIAL)) === null || _buckets$find6 === void 0 ? void 0 : _buckets$find6.doc_count) !== null && _buckets$find$doc_cou6 !== void 0 ? _buckets$find$doc_cou6 : 0,
      [_constants.RuleTranslationResult.UNTRANSLATABLE]: (_buckets$find$doc_cou7 = (_buckets$find7 = buckets.find(({
        key
      }) => key === _constants.RuleTranslationResult.UNTRANSLATABLE)) === null || _buckets$find7 === void 0 ? void 0 : _buckets$find7.doc_count) !== null && _buckets$find$doc_cou7 !== void 0 ? _buckets$find$doc_cou7 : 0
    };
  }
  getFilterQuery(migrationId, filters = {}) {
    var _filters$searchTerm;
    const filter = [{
      term: {
        migration_id: migrationId
      }
    }];
    if (filters.status) {
      if (Array.isArray(filters.status)) {
        filter.push({
          terms: {
            status: filters.status
          }
        });
      } else {
        filter.push({
          term: {
            status: filters.status
          }
        });
      }
    }
    if (filters.ids) {
      filter.push({
        terms: {
          _id: filters.ids
        }
      });
    }
    if ((_filters$searchTerm = filters.searchTerm) !== null && _filters$searchTerm !== void 0 && _filters$searchTerm.length) {
      filter.push(_search.conditions.matchTitle(filters.searchTerm));
    }
    if (filters.installed === true) {
      filter.push(_search.conditions.isInstalled());
    } else if (filters.installed === false) {
      filter.push(_search.conditions.isNotInstalled());
    }
    if (filters.installable === true) {
      filter.push(..._search.conditions.isInstallable());
    } else if (filters.installable === false) {
      filter.push(..._search.conditions.isNotInstallable());
    }
    if (filters.prebuilt === true) {
      filter.push(_search.conditions.isPrebuilt());
    } else if (filters.prebuilt === false) {
      filter.push(_search.conditions.isCustom());
    }
    if (filters.failed === true) {
      filter.push(_search.conditions.isFailed());
    } else if (filters.failed === false) {
      filter.push(_search.conditions.isNotFailed());
    }
    if (filters.fullyTranslated === true) {
      filter.push(_search.conditions.isFullyTranslated());
    } else if (filters.fullyTranslated === false) {
      filter.push(_search.conditions.isNotFullyTranslated());
    }
    if (filters.partiallyTranslated === true) {
      filter.push(_search.conditions.isPartiallyTranslated());
    } else if (filters.partiallyTranslated === false) {
      filter.push(_search.conditions.isNotPartiallyTranslated());
    }
    if (filters.untranslatable === true) {
      filter.push(_search.conditions.isUntranslatable());
    } else if (filters.untranslatable === false) {
      filter.push(_search.conditions.isNotUntranslatable());
    }
    return {
      bool: {
        filter
      }
    };
  }

  /**
   *
   * Prepares bulk ES delete operations for the rules based on migrationId.
   *
   * */
  async prepareDelete(migrationId) {
    const index = await this.getIndexName();
    const rulesToBeDeleted = await this.get(migrationId, {
      size: _constants2.MAX_ES_SEARCH_SIZE
    });
    const rulesToBeDeletedDocIds = rulesToBeDeleted.data.map(rule => rule.id);
    return rulesToBeDeletedDocIds.map(docId => ({
      delete: {
        _id: docId,
        _index: index
      }
    }));
  }
}
exports.RuleMigrationsDataRulesClient = RuleMigrationsDataRulesClient;