"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectValuesProcessor = void 0;
var _painless = require("../painless");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A field retention operator that collects up to `maxLength` values of the specified field.
 * Values are first collected from the field, then from the enrich field if the field is not present or empty.
 */
const collectValuesProcessor = ({
  destination,
  retention
}, {
  enrichField
}) => {
  if ((retention === null || retention === void 0 ? void 0 : retention.operation) !== 'collect_values') {
    throw new Error('Wrong operation for collectValuesProcessor');
  }
  const ctxField = `ctx.${destination}`;
  const ctxEnrichField = `ctx.${enrichField}.${destination}`;
  return {
    script: {
      lang: 'painless',
      source: `
  Set uniqueVals = new HashSet();

  if (!(${(0, _painless.isFieldMissingOrEmpty)(ctxField)})) {
    if(${ctxField} instanceof Collection) {
      uniqueVals.addAll(${ctxField});
    } else {
      uniqueVals.add(${ctxField});
    }
  }

  if (uniqueVals.size() < params.max_length && !(${(0, _painless.isFieldMissingOrEmpty)(ctxEnrichField)})) {
    int remaining = params.max_length - uniqueVals.size();
    List historicalVals = ${ctxEnrichField}.subList(0, (int) Math.min(remaining, ${ctxEnrichField}.size()));
    uniqueVals.addAll(historicalVals);
  }

  ${ctxField} = new ArrayList(uniqueVals).subList(0, (int) Math.min(params.max_length, uniqueVals.size()));
  `,
      params: {
        max_length: retention.maxLength
      }
    }
  };
};
exports.collectValuesProcessor = collectValuesProcessor;