"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUrlState = exports.decodeRisonUrlState = void 0;
var _queryString = require("query-string");
var _react = require("react");
var _rison = require("@kbn/rison");
var _reactRouterDom = require("react-router-dom");
var _common = require("@kbn/kibana-utils-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useUrlState = ({
  defaultState,
  decodeUrlState,
  encodeUrlState,
  urlStateKey,
  writeDefaultState = false
}) => {
  const history = (0, _reactRouterDom.useHistory)();

  // history.location is mutable so we can't reliably use useMemo
  const queryString = history !== null && history !== void 0 && history.location ? getQueryStringFromLocation(history.location) : '';
  const urlStateString = (0, _react.useMemo)(() => {
    if (!queryString) {
      return;
    }
    return getParamFromQueryString(queryString, urlStateKey);
  }, [queryString, urlStateKey]);
  const decodedState = (0, _react.useMemo)(() => {
    return decodeUrlState(decodeRisonUrlState(urlStateString));
  }, [decodeUrlState, urlStateString]);
  const state = (0, _react.useMemo)(() => {
    return typeof decodedState !== 'undefined' ? decodedState : defaultState;
  }, [defaultState, decodedState]);
  const setState = (0, _react.useCallback)(patch => {
    var _decodeUrlState;
    if (!history || !history.location) {
      return;
    }
    const currentLocation = history.location;
    const newState = patch instanceof Function ? patch((_decodeUrlState = decodeUrlState(decodeRisonUrlState(getParamFromQueryString(getQueryStringFromLocation(currentLocation), urlStateKey)))) !== null && _decodeUrlState !== void 0 ? _decodeUrlState : defaultState) : patch;
    const newLocation = replaceQueryStringInLocation(currentLocation, replaceStateKeyInQueryString(urlStateKey, typeof newState !== 'undefined' ? encodeUrlState(newState) : undefined)(getQueryStringFromLocation(currentLocation)));
    if (newLocation !== currentLocation) {
      history.replace(newLocation);
    }
  }, [decodeUrlState, defaultState, encodeUrlState, history, urlStateKey]);
  const [shouldInitialize, setShouldInitialize] = (0, _react.useState)(writeDefaultState && typeof decodedState === 'undefined');
  (0, _react.useEffect)(() => {
    if (shouldInitialize) {
      setShouldInitialize(false);
      setState(defaultState);
    }
  }, [shouldInitialize, setState, defaultState]);
  return [state, setState];
};
exports.useUrlState = useUrlState;
const decodeRisonUrlState = value => {
  try {
    return value ? (0, _rison.decode)(value) : undefined;
  } catch (error) {
    if (error instanceof Error && error.message.startsWith('rison decoder error')) {
      return {};
    }
    throw error;
  }
};
exports.decodeRisonUrlState = decodeRisonUrlState;
const getQueryStringFromLocation = location => location.search.substring(1);
const getParamFromQueryString = (queryString, key) => {
  const parsedQueryString = (0, _queryString.parse)(queryString, {
    sort: false
  });
  const queryParam = parsedQueryString[key];
  return Array.isArray(queryParam) ? queryParam[0] : queryParam;
};
const replaceQueryStringInLocation = (location, queryString) => {
  if (queryString === getQueryStringFromLocation(location)) {
    return location;
  } else {
    return {
      ...location,
      search: `?${queryString}`
    };
  }
};
const encodeRisonUrlState = state => (0, _rison.encode)(state);
const replaceStateKeyInQueryString = (stateKey, urlState) => queryString => {
  const previousQueryValues = (0, _queryString.parse)(queryString, {
    sort: false
  });
  const newValue = typeof urlState === 'undefined' ? previousQueryValues : {
    ...previousQueryValues,
    [stateKey]: encodeRisonUrlState(urlState)
  };
  return (0, _queryString.stringify)(_common.url.encodeQuery(newValue), {
    sort: false,
    encode: false
  });
};