"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeSuggestionWithVisContext = mergeSuggestionWithVisContext;
exports.switchVisualizationType = switchVisualizationType;
var _visualizationUtils = require("@kbn/visualization-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns the suggestion updated with external visualization state for ES|QL charts
 * The visualization state is merged with the suggestion if the datasource is textBased, the columns match the context and the visualization type matches
 * @param suggestion the suggestion to be updated
 * @param visAttributes the preferred visualization attributes
 * @param context the lens suggestions api context as being set by the consumers
 * @returns updated suggestion
 */

function mergeSuggestionWithVisContext({
  suggestion,
  visAttributes,
  context
}) {
  if (visAttributes.visualizationType !== suggestion.visualizationId || !('textBasedColumns' in context)) {
    return suggestion;
  }

  // it should be one of 'formBased'/'textBased' and have value
  const datasourceId = (0, _visualizationUtils.getDatasourceId)(visAttributes.state.datasourceStates);

  // if the datasource is formBased, we should not merge
  if (!datasourceId || datasourceId === 'formBased') {
    return suggestion;
  }
  const datasourceState = Object.assign({}, visAttributes.state.datasourceStates[datasourceId]);

  // should be based on same columns
  if (!(datasourceState !== null && datasourceState !== void 0 && datasourceState.layers) || Object.values(datasourceState === null || datasourceState === void 0 ? void 0 : datasourceState.layers).some(layer => {
    var _layer$columns, _layer$columns2, _context$textBasedCol2;
    return ((_layer$columns = layer.columns) === null || _layer$columns === void 0 ? void 0 : _layer$columns.some(c => {
      var _context$textBasedCol;
      return !(context !== null && context !== void 0 && (_context$textBasedCol = context.textBasedColumns) !== null && _context$textBasedCol !== void 0 && _context$textBasedCol.find(col => col.id === c.fieldName));
    })) || ((_layer$columns2 = layer.columns) === null || _layer$columns2 === void 0 ? void 0 : _layer$columns2.length) !== (context === null || context === void 0 ? void 0 : (_context$textBasedCol2 = context.textBasedColumns) === null || _context$textBasedCol2 === void 0 ? void 0 : _context$textBasedCol2.length);
  })) {
    return suggestion;
  }
  const layerIds = Object.keys(datasourceState.layers);
  try {
    return {
      title: visAttributes.title,
      visualizationId: visAttributes.visualizationType,
      visualizationState: visAttributes.state.visualization,
      keptLayerIds: layerIds,
      datasourceState,
      datasourceId,
      columns: suggestion.columns,
      changeType: suggestion.changeType,
      score: suggestion.score,
      previewIcon: suggestion.previewIcon
    };
  } catch {
    return suggestion;
  }
}

/**
 * Switches the visualization type of a suggestion to the specified visualization type
 * @param visualizationMap the visualization map
 * @param targetTypeId the target visualization type to switch to
 * @param familyType the family type of the current suggestion
 * @param shouldSwitch whether the visualization type should be switched
 * @returns updated suggestion or undefined if no switch was made
 */
function switchVisualizationType({
  visualizationMap,
  suggestions,
  targetTypeId,
  familyType,
  shouldSwitch
}) {
  const suggestion = suggestions.find(s => s.visualizationId === familyType);
  if (shouldSwitch && suggestion && familyType && targetTypeId) {
    var _visualizationMap$sug, _visualizationMap$sug2;
    const visualizationState = (_visualizationMap$sug = visualizationMap[suggestion.visualizationId]) === null || _visualizationMap$sug === void 0 ? void 0 : (_visualizationMap$sug2 = _visualizationMap$sug.switchVisualizationType) === null || _visualizationMap$sug2 === void 0 ? void 0 : _visualizationMap$sug2.call(_visualizationMap$sug, targetTypeId, suggestion === null || suggestion === void 0 ? void 0 : suggestion.visualizationState);
    return [{
      ...suggestion,
      visualizationState
    }];
  }
}