"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTracedEsClient = createTracedEsClient;
var _apmUtils = require("@kbn/apm-utils");
var _lib = require("@kbn/task-manager-plugin/server/metrics/lib");
var _esql_result_to_plain_objects = require("./esql_result_to_plain_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * An Elasticsearch Client with a fully typed `search` method and built-in
 * APM instrumentation.
 */

function createTracedEsClient({
  client,
  logger,
  plugin,
  labels
}) {
  // wraps the ES calls in a named APM span for better analysis
  // (otherwise it would just eg be a _search span)
  const callWithLogger = (operationName, request, callback) => {
    logger.debug(() => `Request (${operationName}):\n${JSON.stringify(request)}`);
    return (0, _apmUtils.withSpan)({
      name: operationName,
      labels: {
        ...labels,
        ...(plugin ? {
          plugin
        } : {})
      }
    }, callback, logger).then(response => {
      logger.trace(() => `Response (${operationName}):\n${JSON.stringify(response, null, 2)}`);
      return response;
    });
  };
  return {
    client,
    fieldCaps(operationName, parameters) {
      return callWithLogger(operationName, parameters, () => {
        return client.fieldCaps({
          ...parameters
        });
      });
    },
    esql(operationName, parameters, options) {
      return callWithLogger(operationName, parameters, () => {
        return client.esql.query({
          ...parameters
        }, {
          querystring: {
            drop_null_columns: true
          }
        }).then(response => {
          var _options$transform;
          const esqlResponse = response;
          const transform = (_options$transform = options === null || options === void 0 ? void 0 : options.transform) !== null && _options$transform !== void 0 ? _options$transform : 'none';
          if (transform === 'none') {
            return esqlResponse;
          }
          const parsedResponse = {
            hits: (0, _esql_result_to_plain_objects.esqlResultToPlainObjects)(esqlResponse)
          };
          if (transform === 'plain') {
            return parsedResponse;
          }
          return {
            hits: parsedResponse.hits.map(hit => (0, _lib.unflattenObject)(hit))
          };
        });
      });
    },
    search(operationName, parameters) {
      return callWithLogger(operationName, parameters, () => {
        return client.search(parameters);
      });
    },
    msearch(operationName, parameters) {
      return callWithLogger(operationName, parameters, () => {
        return client.msearch(parameters);
      });
    }
  };
}