"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AddEditMonitorAPI = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _common = require("@kbn/fleet-plugin/common");
var _i18n = require("@kbn/i18n");
var _saved_objects = require("../../../../common/types/saved_objects");
var _delete_monitor_api = require("../services/delete_monitor_api");
var _utils = require("./utils");
var _monitor_validation = require("../monitor_validation");
var _common2 = require("../../common");
var _monitor_management = require("../../../../common/constants/monitor_management");
var _runtime_types = require("../../../../common/runtime_types");
var _common_fields = require("../../../synthetics_service/project_monitor/normalizers/common_fields");
var _monitor_defaults = require("../../../../common/constants/monitor_defaults");
var _test_now_monitor = require("../../synthetics_service/test_now_monitor");
var _default_alert_service = require("../../default_alerts/default_alert_service");
var _monitor_upgrade_sender = require("../../telemetry/monitor_upgrade_sender");
var _formatters = require("../../../../common/formatters");
var _get_private_locations = require("../../../synthetics_service/get_private_locations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AddEditMonitorAPI {
  constructor(routeContext) {
    (0, _defineProperty2.default)(this, "routeContext", void 0);
    (0, _defineProperty2.default)(this, "allPrivateLocations", void 0);
    (0, _defineProperty2.default)(this, "setupGettingStarted", configId => {
      const {
        server,
        request
      } = this.routeContext;
      try {
        const {
          gettingStarted
        } = request.query;
        if (gettingStarted) {
          // ignore await, since we don't want to block the response
          (0, _test_now_monitor.triggerTestNow)(configId, this.routeContext).then(() => {
            server.logger.debug(`Successfully triggered test for monitor: ${configId}`);
          }).catch(error => {
            server.logger.error(`Error triggering test for monitor: ${configId}, Error: ${error.message}`, {
              error
            });
          });
        }
      } catch (error) {
        server.logger.error(`Error triggering test for getting started monitor: ${configId}`, {
          error
        });
      }
    });
    this.routeContext = routeContext;
  }
  async syncNewMonitor({
    id,
    normalizedMonitor,
    savedObjectType
  }) {
    const {
      server,
      syntheticsMonitorClient,
      spaceId
    } = this.routeContext;
    const newMonitorId = id !== null && id !== void 0 ? id : (0, _uuid.v4)();
    let monitorSavedObject = null;
    const monitorWithNamespace = this.hydrateMonitorFields({
      normalizedMonitor,
      newMonitorId
    });
    try {
      var _this$allPrivateLocat, _packagePolicyResult$, _packagePolicyResult$2;
      const newMonitorPromise = this.routeContext.monitorConfigRepository.create({
        normalizedMonitor: monitorWithNamespace,
        id: newMonitorId,
        spaceId,
        savedObjectType
      });
      const syncErrorsPromise = syntheticsMonitorClient.addMonitors([{
        monitor: monitorWithNamespace,
        id: newMonitorId
      }], (_this$allPrivateLocat = this.allPrivateLocations) !== null && _this$allPrivateLocat !== void 0 ? _this$allPrivateLocat : [], spaceId);
      const [monitorSavedObjectN, [packagePolicyResult, syncErrors]] = await Promise.all([newMonitorPromise, syncErrorsPromise]);
      if (packagePolicyResult && ((_packagePolicyResult$ = packagePolicyResult === null || packagePolicyResult === void 0 ? void 0 : (_packagePolicyResult$2 = packagePolicyResult.failed) === null || _packagePolicyResult$2 === void 0 ? void 0 : _packagePolicyResult$2.length) !== null && _packagePolicyResult$ !== void 0 ? _packagePolicyResult$ : []) > 0) {
        const failed = packagePolicyResult.failed.map(f => f.error);
        throw new Error(failed.join(', '));
      }
      monitorSavedObject = monitorSavedObjectN;
      (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryEvent)({
        errors: syncErrors,
        monitor: monitorSavedObject,
        isInlineScript: Boolean(normalizedMonitor[_monitor_management.ConfigKey.SOURCE_INLINE]),
        stackVersion: server.stackVersion
      }));
      return {
        errors: syncErrors,
        newMonitor: {
          ...monitorSavedObject,
          attributes: {
            ...monitorWithNamespace,
            ...monitorSavedObject.attributes
          }
        }
      };
    } catch (e) {
      e.message = `${e.message}, monitor name: ${monitorWithNamespace[_monitor_management.ConfigKey.NAME]}`;
      await this.revertMonitorIfCreated({
        newMonitorId
      });
      throw e;
    }
  }
  validateMonitorType(monitorFields, previousMonitor) {
    const {
      [_monitor_management.ConfigKey.MONITOR_TYPE]: monitorType
    } = monitorFields;
    if (previousMonitor) {
      const {
        [_monitor_management.ConfigKey.MONITOR_TYPE]: prevMonitorType
      } = previousMonitor;
      if (monitorType !== prevMonitorType) {
        // monitor type cannot be changed
        throw new _monitor_validation.MonitorValidationError({
          valid: false,
          reason: _i18n.i18n.translate('xpack.synthetics.createMonitor.validation.monitorTypeChanged', {
            defaultMessage: 'Monitor type cannot be changed from {prevMonitorType} to {monitorType}.',
            values: {
              prevMonitorType,
              monitorType
            }
          }),
          details: '',
          payload: monitorFields
        });
      }
    }
  }
  async normalizeMonitor(requestPayload, monitorPayload, prevLocations) {
    var _request$query;
    const {
      savedObjectsClient,
      syntheticsMonitorClient,
      request
    } = this.routeContext;
    const internal = Boolean((_request$query = request.query) === null || _request$query === void 0 ? void 0 : _request$query.internal);
    const {
      locations,
      private_locations: privateLocations,
      schedule,
      retest_on_failure: retestOnFailure,
      url: rawUrl,
      host: rawHost,
      ...rest
    } = requestPayload;
    const monitor = rest;
    const monitorType = monitor[_monitor_management.ConfigKey.MONITOR_TYPE];
    if (monitorType === _runtime_types.MonitorTypeEnum.HTTP && !monitor.name) {
      monitor.name = monitor.urls;
    }
    const defaultFields = _monitor_defaults.DEFAULT_FIELDS[monitorType];
    let locationsVal = [];
    if (!locations && !privateLocations && prevLocations) {
      locationsVal = prevLocations;
    } else {
      const monitorLocations = (0, _utils.parseMonitorLocations)(monitorPayload, prevLocations, internal);
      if (monitorLocations.privateLocations.length > 0) {
        this.allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(savedObjectsClient);
      } else {
        this.allPrivateLocations = [];
      }
      locationsVal = (0, _common_fields.getMonitorLocations)({
        monitorLocations,
        allPublicLocations: syntheticsMonitorClient.syntheticsService.locations,
        allPrivateLocations: this.allPrivateLocations
      });
    }
    return {
      ..._monitor_defaults.DEFAULT_FIELDS[monitorType],
      ...monitor,
      [_monitor_management.ConfigKey.SCHEDULE]: (0, _common_fields.getMonitorSchedule)(schedule !== null && schedule !== void 0 ? schedule : defaultFields[_monitor_management.ConfigKey.SCHEDULE]),
      [_monitor_management.ConfigKey.MAX_ATTEMPTS]: (0, _common_fields.getMaxAttempts)(retestOnFailure, monitor[_monitor_management.ConfigKey.MAX_ATTEMPTS]),
      [_monitor_management.ConfigKey.LOCATIONS]: locationsVal
    };
  }
  async validateUniqueMonitorName(name, id) {
    const {
      monitorConfigRepository
    } = this.routeContext;
    const kqlFilter = (0, _common2.getSavedObjectKqlFilter)({
      field: 'name.keyword',
      values: name
    });
    const {
      total
    } = await monitorConfigRepository.find({
      perPage: 0,
      filter: id ? `${kqlFilter} and not (${_saved_objects.syntheticsMonitorAttributes}.config_id: ${id})` : kqlFilter
    });
    if (total > 0) {
      return _i18n.i18n.translate('xpack.synthetics.createMonitor.validation.uniqueName', {
        defaultMessage: 'Monitor name must be unique, "{name}" already exists.',
        values: {
          name
        }
      });
    }
  }
  initDefaultAlerts(name) {
    const {
      server,
      savedObjectsClient,
      context,
      request
    } = this.routeContext;
    const {
      gettingStarted
    } = request.query;
    if (!gettingStarted) {
      return;
    }
    try {
      // we do this async, so we don't block the user, error handling will be done on the UI via separate api
      const defaultAlertService = new _default_alert_service.DefaultAlertService(context, server, savedObjectsClient);
      defaultAlertService.setupDefaultAlerts().then(() => {
        server.logger.debug(`Successfully created default alert for monitor: ${name}`);
      }).catch(error => {
        server.logger.error(`Error creating default alert: ${error} for monitor: ${name}`, {
          error
        });
      });
    } catch (error) {
      server.logger.error(`Error creating default alert: ${error} for monitor: ${name}`, {
        error
      });
    }
  }
  hydrateMonitorFields({
    newMonitorId,
    normalizedMonitor
  }) {
    const {
      request
    } = this.routeContext;
    const {
      preserve_namespace: preserveNamespace
    } = request.query;
    return {
      ...normalizedMonitor,
      [_monitor_management.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_monitor_management.ConfigKey.CUSTOM_HEARTBEAT_ID] || newMonitorId,
      [_monitor_management.ConfigKey.CONFIG_ID]: newMonitorId,
      [_monitor_management.ConfigKey.NAMESPACE]: preserveNamespace ? normalizedMonitor[_monitor_management.ConfigKey.NAMESPACE] : this.getMonitorNamespace(normalizedMonitor[_monitor_management.ConfigKey.NAMESPACE])
    };
  }
  getMonitorNamespace(configuredNamespace) {
    const {
      spaceId
    } = this.routeContext;
    const kibanaNamespace = (0, _formatters.formatKibanaNamespace)(spaceId);
    const namespace = configuredNamespace === _monitor_defaults.DEFAULT_NAMESPACE_STRING ? kibanaNamespace : configuredNamespace;
    const {
      error
    } = (0, _common.isValidNamespace)(namespace);
    if (error) {
      throw new Error(`Cannot save monitor. Monitor namespace is invalid: ${error}`);
    }
    return namespace;
  }
  async revertMonitorIfCreated({
    newMonitorId
  }) {
    const {
      server,
      monitorConfigRepository
    } = this.routeContext;
    try {
      const encryptedMonitor = await monitorConfigRepository.get(newMonitorId);
      if (encryptedMonitor) {
        await monitorConfigRepository.bulkDelete([{
          id: newMonitorId,
          type: _saved_objects.syntheticsMonitorSavedObjectType
        }, {
          id: newMonitorId,
          type: _saved_objects.legacySyntheticsMonitorTypeSingle
        }]);
        const deleteMonitorAPI = new _delete_monitor_api.DeleteMonitorAPI(this.routeContext);
        await deleteMonitorAPI.execute({
          monitorIds: [newMonitorId]
        });
      }
    } catch (error) {
      // ignore errors here
      server.logger.error(`Unable to revert monitor with id ${newMonitorId}, Error: ${error.message}`, {
        error
      });
    }
  }
}
exports.AddEditMonitorAPI = AddEditMonitorAPI;