"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TrieNode = exports.ProbabilisticTrie = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint max-classes-per-file: ["error", 2] */

class ProbabilisticTrie {
  constructor() {
    (0, _defineProperty2.default)(this, "root", new TrieNode(null));
  }
  insert(chunks) {
    let currentNode = this.root;
    for (const chunk of chunks) {
      if (!currentNode.children.has(chunk)) {
        currentNode.children.set(chunk, new TrieNode(currentNode));
      }
      currentNode = currentNode.children.get(chunk);
      currentNode.count += 1;
    }
    this.root.count += 1;
  }
  *[Symbol.iterator]() {
    yield* this.root;
  }
  static fromArray(values) {
    const trie = new ProbabilisticTrie();
    values.forEach(chunks => trie.insert(chunks));
    return trie;
  }
}

/**
 * A tuple containing: level of node in tree, message chunks, probability of occurrence, node object.
 */
exports.ProbabilisticTrie = ProbabilisticTrie;
class TrieNode {
  constructor(parent) {
    (0, _defineProperty2.default)(this, "count", 0);
    (0, _defineProperty2.default)(this, "children", new Map());
    this.parent = parent;
  }
  isRoot() {
    return this.parent === null;
  }
  isLeaf() {
    return this.children.size === 0;
  }
  getProbability() {
    if (this.parent === null) {
      return 1;
    }
    return this.count / this.parent.count;
  }
  [Symbol.iterator]() {
    return this.getChunks(0, [], 1);
  }
  isRelevant(parentChunks) {
    if (!parentChunks.length) {
      return false;
    }
    if (this.children.size === 1) {
      const onlyChild = getFirst(this.children);
      return this.count !== onlyChild.count;
    }
    return true;
  }
  *getChunks(parentLevel, parentChunks, parentProbability) {
    const isRelevant = this.isRelevant(parentChunks);
    if (isRelevant) {
      yield [parentLevel, parentChunks, parentProbability, this];
    }
    for (const [chunk, child] of this.children) {
      yield* child.getChunks(isRelevant ? parentLevel + 1 : parentLevel, [...parentChunks, chunk], parentProbability * child.getProbability());
    }
  }
}
exports.TrieNode = TrieNode;
function getFirst(map) {
  const result = map.values().next();
  if (!result.done) {
    return result.value;
  }
}