"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SpacesPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _analytics = require("./analytics");
var _create_feature_catalogue_entry = require("./create_feature_catalogue_entry");
var _management = require("./management");
var _nav_control = require("./nav_control");
var _space_selector = require("./space_selector");
var _spaces_manager = require("./spaces_manager");
var _ui_api = require("./ui_api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Setup contract for the Spaces plugin.
 */

/**
 * Start contract for the Spaces plugin.
 */

class SpacesPlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "spacesManager", void 0);
    (0, _defineProperty2.default)(this, "spacesApi", void 0);
    (0, _defineProperty2.default)(this, "eventTracker", void 0);
    (0, _defineProperty2.default)(this, "managementService", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    this.initializerContext = initializerContext;
    this.config = this.initializerContext.config.get();
    this.isServerless = this.initializerContext.env.packageInfo.buildFlavor === 'serverless';
  }
  setup(core, plugins) {
    if (this.config.allowSolutionVisibility === undefined) {
      throw new Error('allowSolutionVisibility has not been set in the Spaces plugin config.');
    }
    const hasOnlyDefaultSpace = this.config.maxSpaces === 1;
    this.spacesManager = new _spaces_manager.SpacesManager(core.http);
    this.spacesApi = {
      ui: (0, _ui_api.getUiApi)({
        spacesManager: this.spacesManager,
        getStartServices: core.getStartServices
      }),
      getActiveSpace$: () => this.spacesManager.onActiveSpaceChange$,
      getActiveSpace: () => this.spacesManager.getActiveSpace(),
      hasOnlyDefaultSpace,
      isSolutionViewEnabled: this.config.allowSolutionVisibility
    };
    (0, _analytics.registerSpacesEventTypes)(core);
    this.eventTracker = new _analytics.EventTracker(core.analytics);

    // Only skip setup of space selector and management service if serverless and only one space is allowed
    if (!(this.isServerless && hasOnlyDefaultSpace)) {
      const getIsRoleManagementEnabled = async () => {
        const {
          security
        } = await core.plugins.onSetup('security');
        if (!security.found) {
          throw new Error('Security plugin is not available as runtime dependency.');
        }
        return security.contract.authz.isRoleManagementEnabled;
      };
      const getRolesAPIClient = async () => {
        const {
          security
        } = await core.plugins.onSetup('security');
        if (!security.found) {
          throw new Error('Security plugin is not available as runtime dependency.');
        }
        return security.contract.authz.roles;
      };
      const getPrivilegesAPIClient = async () => {
        const {
          security
        } = await core.plugins.onSetup('security');
        if (!security.found) {
          throw new Error('Security plugin is not available as runtime dependency.');
        }
        return security.contract.authz.privileges;
      };
      const getSecurityLicense = async () => {
        const {
          security
        } = await core.plugins.onSetup('security');
        if (!security.found) {
          throw new Error('Security plugin is not available as runtime dependency.');
        }
        return security.contract.license;
      };
      if (plugins.home) {
        plugins.home.featureCatalogue.register((0, _create_feature_catalogue_entry.createSpacesFeatureCatalogueEntry)());
      }
      if (plugins.management) {
        this.managementService = new _management.ManagementService();
        this.managementService.setup({
          management: plugins.management,
          getStartServices: core.getStartServices,
          spacesManager: this.spacesManager,
          config: this.config,
          logger: this.initializerContext.logger.get(),
          getIsRoleManagementEnabled,
          getRolesAPIClient,
          eventTracker: this.eventTracker,
          getPrivilegesAPIClient,
          isServerless: this.isServerless,
          getSecurityLicense
        });
      }
      _space_selector.spaceSelectorApp.create({
        getStartServices: core.getStartServices,
        application: core.application,
        spacesManager: this.spacesManager
      });
    }
    (0, _analytics.registerAnalyticsContext)(core.analytics, this.spacesManager.onActiveSpaceChange$);
    return {
      hasOnlyDefaultSpace,
      isSolutionViewEnabled: this.config.allowSolutionVisibility
    };
  }
  start(core) {
    // Only skip spaces navigation if serverless and only one space is allowed
    if (!(this.isServerless && this.config.maxSpaces === 1)) {
      (0, _nav_control.initSpacesNavControl)(this.spacesManager, core, this.config, this.eventTracker);
    }
    return this.spacesApi;
  }
  stop() {
    if (this.managementService) {
      this.managementService.stop();
      this.managementService = undefined;
    }
  }
}
exports.SpacesPlugin = SpacesPlugin;