"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TYPE = exports.BulkDeleteTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _delete_slo = require("../../delete_slo");
var _slo_repository = require("../../slo_repository");
var _summary_transform_generator = require("../../summary_transform_generator/summary_transform_generator");
var _summay_transform_manager = require("../../summay_transform_manager");
var _transform_manager = require("../../transform_manager");
var _run_bulk_delete = require("./run_bulk_delete");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'slo:bulk-delete-task';
class BulkDeleteTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "logger", void 0);
    const {
      core,
      plugins,
      logFactory
    } = setupContract;
    this.logger = logFactory.get(TYPE);
    this.logger.debug('Registering task with [10m] timeout');
    plugins.taskManager.setup.registerTaskDefinitions({
      [TYPE]: {
        title: 'SLO bulk delete',
        timeout: '10m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance,
          fakeRequest
        }) => {
          return {
            run: async () => {
              this.logger.debug(`starting bulk delete operation`);
              if (!fakeRequest) {
                this.logger.debug('fakeRequest is not defined');
                return;
              }
              const state = taskInstance.state;
              if (state.isDone) {
                // The task was done in the previous run,
                // we only rescheduled it once for keeping an ephemeral state for the user
                return;
              }
              this.abortController = new AbortController();
              const [coreStart, pluginStart] = await core.getStartServices();
              const scopedClusterClient = coreStart.elasticsearch.client.asScoped(fakeRequest);
              const scopedSoClient = coreStart.savedObjects.getScopedClient(fakeRequest);
              const rulesClient = await pluginStart.alerting.getRulesClientWithRequest(fakeRequest);
              const repository = new _slo_repository.KibanaSavedObjectsSLORepository(scopedSoClient, this.logger);
              const transformManager = new _transform_manager.DefaultTransformManager({}, scopedClusterClient, this.logger, this.abortController);
              const summaryTransformManager = new _summay_transform_manager.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), scopedClusterClient, this.logger, this.abortController);
              const deleteSLO = new _delete_slo.DeleteSLO(repository, transformManager, summaryTransformManager, scopedClusterClient, rulesClient, this.abortController);
              try {
                const params = taskInstance.params;
                const results = await (0, _run_bulk_delete.runBulkDelete)(params, {
                  deleteSLO,
                  scopedClusterClient,
                  rulesClient,
                  logger: this.logger,
                  abortController: this.abortController
                });
                return {
                  runAt: new Date(Date.now() + 60 * 60 * 1000),
                  state: {
                    isDone: true,
                    results
                  }
                };
              } catch (err) {
                this.logger.debug(`Error: ${err}`);
                return {
                  runAt: new Date(Date.now() + 60 * 60 * 1000),
                  state: {
                    isDone: true,
                    error: err.message
                  }
                };
              }
            },
            cancel: async () => {
              this.abortController.abort('Timed out');
            }
          };
        }
      }
    });
  }
}
exports.BulkDeleteTask = BulkDeleteTask;