"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildSpaceDataFilter = void 0;
var _stringify = require("../../../endpoint/utils/stringify");
var _artifacts = require("../../../../common/endpoint/service/artifacts");
var _utils = require("../../../../common/endpoint/service/artifacts/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Builds the KQL string for filtering artifacts that can be accessed from active space
 * @param endpointServices
 * @param httpRequest
 */
const buildSpaceDataFilter = async (endpointServices, httpRequest) => {
  const logger = endpointServices.createLogger('buildSpaceDataFilter');
  const spaceId = (await endpointServices.getActiveSpace(httpRequest)).id;
  const fleetServices = endpointServices.getInternalFleetServices(spaceId);
  const soScopedClient = fleetServices.savedObjects.createInternalScopedSoClient({
    spaceId
  });
  const {
    items: allEndpointPolicyIds
  } = await fleetServices.packagePolicy.listIds(soScopedClient, {
    kuery: fleetServices.endpointPolicyKuery,
    perPage: 10_000
  });
  logger.debug(() => `policies currently visible in space ID [${spaceId}]:\n${(0, _stringify.stringify)(allEndpointPolicyIds)}`);

  // Filter to scope down the data visible in active space id:
  //
  //      (
  //         All global artifacts
  //         -OR-
  //         All per-policy artifacts assigned to a policy visible in active space
  //      )
  //      -OR-
  //      (
  //         Artifacts NOT containing a `policy:` tag ("dangling" per-policy artifacts)
  //         -AND-
  //         having an owner space ID value that matches active space
  //      )
  //
  const spaceVisibleDataFilter = `
      (
        (
          exception-list-agnostic.attributes.tags:("${_artifacts.GLOBAL_ARTIFACT_TAG}"${allEndpointPolicyIds.length === 0 ? ')' : ` OR ${allEndpointPolicyIds.map(policyId => `"${(0, _utils.buildPerPolicyTag)(policyId)}"`).join(' OR ')}
          )`}
        )
        OR
        (
          NOT exception-list-agnostic.attributes.tags:"${(0, _utils.buildPerPolicyTag)('*')}"
          AND
          exception-list-agnostic.attributes.tags:"${(0, _utils.buildSpaceOwnerIdTag)(spaceId)}"
        )
      )`;
  logger.debug(`Filter for space id [${spaceId}]:\n${spaceVisibleDataFilter}`);
  return {
    filter: spaceVisibleDataFilter
  };
};
exports.buildSpaceDataFilter = buildSpaceDataFilter;