"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRetrieveIntegrationsNode = void 0;
var _output_parsers = require("@langchain/core/output_parsers");
var _comments = require("../../../../../util/comments");
var _prompts = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getRetrieveIntegrationsNode = ({
  model,
  ruleMigrationsRetriever,
  telemetryClient
}) => {
  return async state => {
    const query = state.semantic_query;
    const integrations = await ruleMigrationsRetriever.integrations.search(query);
    if (integrations.length === 0) {
      telemetryClient.reportIntegrationsMatch({
        preFilterIntegrations: []
      });
      const comment = '## Integration Matching Summary\n\nNo related integration found.';
      return {
        comments: [(0, _comments.generateAssistantComment)(comment)]
      };
    }
    const outputParser = new _output_parsers.JsonOutputParser();
    const mostRelevantIntegration = _prompts.MATCH_INTEGRATION_PROMPT.pipe(model).pipe(outputParser);
    const integrationsInfo = integrations.map(integration => ({
      title: integration.title,
      description: integration.description
    }));
    const splunkRule = {
      title: state.original_rule.title,
      description: state.original_rule.description
    };

    /*
     * Takes the most relevant integration from the array of integration(s) returned by the semantic query, returns either the most relevant or none.
     */
    const integrationsJson = JSON.stringify(integrationsInfo, null, 2);
    const response = await mostRelevantIntegration.invoke({
      integrations: integrationsJson,
      splunk_rule: JSON.stringify(splunkRule, null, 2)
    });
    const comments = response.summary ? [(0, _comments.generateAssistantComment)((0, _comments.cleanMarkdown)(response.summary))] : undefined;
    if (response.match) {
      const matchedIntegration = integrations.find(r => r.title === response.match);
      telemetryClient.reportIntegrationsMatch({
        preFilterIntegrations: integrations,
        postFilterIntegration: matchedIntegration
      });
      if (matchedIntegration) {
        return {
          integration: matchedIntegration,
          comments
        };
      }
    }
    return {
      comments
    };
  };
};
exports.getRetrieveIntegrationsNode = getRetrieveIntegrationsNode;