"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importRules = void 0;
var _utils = require("../../../routes/utils");
var _errors = require("./errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Takes a stream of rules to be imported and either creates or updates rules
 * based on user overwrite preferences
 * @param ruleChunks {@link RuleToImport} - rules being imported
 * @param overwriteRules {boolean} - whether to overwrite existing rules
 * with imported rules if their rule_id matches
 * @param detectionRulesClient {object}
 * @returns {Promise} an array of error and success messages from import
 */
const importRules = async ({
  ruleChunks,
  overwriteRules,
  detectionRulesClient,
  ruleSourceImporter,
  allowMissingConnectorSecrets
}) => {
  const response = [];
  if (ruleChunks.length === 0) {
    return response;
  }
  for (const rules of ruleChunks) {
    const importedRulesResponse = await detectionRulesClient.importRules({
      allowMissingConnectorSecrets,
      overwriteRules,
      ruleSourceImporter,
      rules
    });
    const importResponses = importedRulesResponse.map(rule => {
      if ((0, _errors.isRuleImportError)(rule)) {
        return (0, _utils.createBulkErrorObject)({
          message: rule.error.message,
          statusCode: (0, _errors.isRuleConflictError)(rule) ? 409 : 400,
          ruleId: rule.error.ruleId
        });
      }
      return {
        rule_id: rule.rule_id,
        status_code: 200
      };
    });
    response.push(...importResponses);
  }
  return response;
};
exports.importRules = importRules;