"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importRules = void 0;
var _i18n = require("@kbn/i18n");
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
var _errors = require("../../import/errors");
var _check_rule_exception_references = require("../../import/check_rule_exception_references");
var _gather_referenced_exceptions = require("../../import/gather_referenced_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Imports rules
 */

const importRules = async ({
  allowMissingConnectorSecrets,
  detectionRulesClient,
  overwriteRules,
  ruleSourceImporter,
  rules,
  savedObjectsClient
}) => {
  const existingLists = await (0, _gather_referenced_exceptions.getReferencedExceptionLists)({
    rules,
    savedObjectsClient
  });
  await ruleSourceImporter.setup(rules);
  return Promise.all(rules.map(async rule => {
    const errors = [];
    try {
      if (!ruleSourceImporter.isPrebuiltRule(rule)) {
        var _rule$version;
        rule.version = (_rule$version = rule.version) !== null && _rule$version !== void 0 ? _rule$version : 1;
      }
      if (!(0, _rule_management.ruleToImportHasVersion)(rule)) {
        return (0, _errors.createRuleImportErrorObject)({
          message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.rules.cannotImportPrebuiltRuleWithoutVersion', {
            defaultMessage: 'Prebuilt rules must specify a "version" to be imported. [rule_id: {ruleId}]',
            values: {
              ruleId: rule.rule_id
            }
          }),
          ruleId: rule.rule_id
        });
      }
      const [exceptionErrors, exceptions] = (0, _check_rule_exception_references.checkRuleExceptionReferences)({
        rule,
        existingLists
      });
      errors.push(...exceptionErrors);
      const {
        immutable,
        ruleSource
      } = ruleSourceImporter.calculateRuleSource(rule);
      const importedRule = await detectionRulesClient.importRule({
        ruleToImport: {
          ...rule,
          exceptions_list: [...exceptions]
        },
        overrideFields: {
          rule_source: ruleSource,
          immutable
        },
        overwriteRules,
        allowMissingConnectorSecrets
      });
      return [...errors, importedRule];
    } catch (err) {
      var _ref;
      const {
        error,
        message
      } = err;
      const caughtError = (0, _errors.isRuleImportError)(err) ? err : (0, _errors.createRuleImportErrorObject)({
        ruleId: rule.rule_id,
        message: (_ref = message !== null && message !== void 0 ? message : error === null || error === void 0 ? void 0 : error.message) !== null && _ref !== void 0 ? _ref : 'unknown error'
      });
      return [...errors, caughtError];
    }
  })).then(results => results.flat());
};
exports.importRules = importRules;