"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SimpleMemCache = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/no-explicit-any */

/**
 * A simple memory caching mechanism. Entries are given a time-to-live (`ttl`) and deleted only when
 * attempted to be retrieved and entry is expired.
 *
 * > **NOTE**:  There is no automated "cache cleanup" to remove expired entries over time due to the
 * >            fact that it could lead to memory leaks. A `cleanup()` method, however, is provided
 * >            which can be called periodically to clean up the cache
 */
class SimpleMemCache {
  constructor({
    ttl = 10
  } = {}) {
    (0, _defineProperty2.default)(this, "ttl", void 0);
    (0, _defineProperty2.default)(this, "cache", new Map());
    this.ttl = ttl;
  }
  isExpired(entry) {
    return entry.expires < Date.now();
  }
  set(key, value, ttl = this.ttl) {
    const expiresDt = new Date();
    expiresDt.setSeconds(expiresDt.getSeconds() + ttl);
    this.cache.set(key, {
      value,
      expires: expiresDt.getTime()
    });
  }
  get(key) {
    const cachedValue = this.cache.get(key);
    if (cachedValue) {
      if (this.isExpired(cachedValue)) {
        this.delete(key);
        return;
      }
      return cachedValue.value;
    }
  }
  delete(key) {
    this.cache.delete(key);
  }
  deleteAll() {
    this.cache.clear();
  }
  cleanup() {
    for (const [cacheKey, cacheData] of this.cache.entries()) {
      if (this.isExpired(cacheData)) {
        this.delete(cacheKey);
      }
    }
  }
}
exports.SimpleMemCache = SimpleMemCache;