"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SECURITY_LABS_KNOWLEDGE_BASE_TOOL = void 0;
var _tools = require("@langchain/core/tools");
var _zod = require("@kbn/zod");
var _constants = require("@kbn/elastic-assistant-plugin/server/routes/knowledge_base/constants");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _jsYaml = _interopRequireDefault(require("js-yaml"));
var _references = require("@kbn/elastic-assistant-common/impl/content_references/references");
var _document = require("langchain/document");
var _helpers = require("@kbn/elastic-assistant-plugin/server/routes/helpers");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolDetails = {
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: 'Call this for knowledge from Elastic Security Labs content, which contains information on malware, attack techniques, and more.',
  id: 'security-labs-knowledge-base-tool',
  name: 'SecurityLabsKnowledgeBaseTool'
};
const SECURITY_LABS_BASE_URL = 'https://www.elastic.co/security-labs/';
const SECURITY_LABS_KNOWLEDGE_BASE_TOOL = exports.SECURITY_LABS_KNOWLEDGE_BASE_TOOL = {
  ...toolDetails,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    const {
      kbDataClient,
      isEnabledKnowledgeBase
    } = params;
    return isEnabledKnowledgeBase && kbDataClient != null;
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      kbDataClient,
      contentReferencesStore
    } = params;
    return (0, _tools.tool)(async input => {
      const docs = await kbDataClient.getKnowledgeBaseDocumentEntries({
        kbResource: _constants.SECURITY_LABS_RESOURCE,
        query: input.question
      });
      if (docs.length === 0) {
        const isKnowledgeBaseInstalled = await (0, _helpers.getIsKnowledgeBaseInstalled)(kbDataClient);
        if (!isKnowledgeBaseInstalled) {
          // prompt to help user install knowledge base
          return 'The "AI Assistant knowledge base" needs to be installed, containing the Security Labs content. Navigate to the Knowledge Base page in the AI Assistant Settings to install it.';
        }
      }
      const citedDocs = docs.map(doc => {
        let reference;
        try {
          const yamlString = doc.pageContent.split('---')[1];
          const parsed = _jsYaml.default.load(yamlString);
          const slug = parsed.slug;
          const title = parsed.title;
          if (!slug || !title) {
            throw new Error('Slug or title not found in YAML');
          }
          reference = contentReferencesStore.add(p => (0, _references.hrefReference)(p.id, `${SECURITY_LABS_BASE_URL}${slug}`, `Security Labs: ${title}`));
        } catch (_error) {
          reference = contentReferencesStore.add(p => (0, _references.knowledgeBaseReference)(p.id, 'Elastic Security Labs content', 'securityLabsId'));
        }
        return new _document.Document({
          id: doc.id,
          pageContent: `${(0, _elasticAssistantCommon.contentReferenceString)(reference)}\n${doc.pageContent}`,
          metadata: doc.metadata
        });
      });

      // TODO: Token pruning
      const result = JSON.stringify(citedDocs).substring(0, 20000);
      return result;
    }, {
      name: toolDetails.name,
      description: params.description || toolDetails.description,
      schema: _zod.z.object({
        question: _zod.z.string().describe(`Key terms to retrieve Elastic Security Labs content for, like specific malware names or attack techniques.`)
      }),
      tags: ['security-labs', 'knowledge-base']
    });
  }
};