"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMCPRoutes = registerMCPRoutes;
var _mcp = require("@modelcontextprotocol/sdk/server/mcp.js");
var _types = require("@modelcontextprotocol/sdk/types.js");
var _configSchema = require("@kbn/config-schema");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
var _kibana_mcp_http_transport = require("../utils/kibana_mcp_http_transport");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TECHNICAL_PREVIEW_WARNING = 'Elastic MCP Server is in technical preview and may be changed or removed in a future release. Elastic will work to fix any issues, but features in technical preview are not subject to the support SLA of official GA features.';
const MCP_SERVER_NAME = 'elastic-mcp-server';
const MCP_SERVER_VERSION = '0.0.1';
const MCP_SERVER_PATH = '/api/mcp';
function registerMCPRoutes({
  router,
  getInternalServices,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });
  router.versioned.post({
    path: MCP_SERVER_PATH,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'MCP server',
    description: TECHNICAL_PREVIEW_WARNING,
    options: {
      tags: ['mcp'],
      xsrfRequired: false,
      availability: {
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: _configSchema.schema.object({}, {
          unknowns: 'allow'
        })
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    let transport;
    let server;
    const {
      uiSettings
    } = await ctx.core;
    const enabled = await uiSettings.client.get(_constants.ONECHAT_MCP_SERVER_UI_SETTING_ID);
    if (!enabled) {
      return response.notFound();
    }
    try {
      transport = new _kibana_mcp_http_transport.KibanaMcpHttpTransport({
        sessionIdGenerator: undefined,
        logger
      });

      // Instantiate new MCP server upon every request, no session persistence
      server = new _mcp.McpServer({
        name: MCP_SERVER_NAME,
        version: MCP_SERVER_VERSION
      });
      const {
        tools: toolService
      } = getInternalServices();
      const registry = toolService.registry.asScopedPublicRegistry({
        request
      });
      const tools = await registry.list({});

      // Expose tools scoped to the request
      for (const tool of tools) {
        server.tool(tool.id, tool.description, tool.schema.shape, async args => {
          const toolResult = await tool.execute({
            toolParams: args
          });
          return {
            content: [{
              type: 'text',
              text: JSON.stringify(toolResult)
            }]
          };
        });
      }
      request.events.aborted$.subscribe(async () => {
        var _transport, _server;
        await ((_transport = transport) === null || _transport === void 0 ? void 0 : _transport.close().catch(error => {
          logger.error('MCP Server: Error closing transport', {
            error
          });
        }));
        await ((_server = server) === null || _server === void 0 ? void 0 : _server.close().catch(error => {
          logger.error('MCP Server: Error closing server', {
            error
          });
        }));
      });
      await server.connect(transport);
      return await transport.handleRequest(request, response);
    } catch (error) {
      logger.error('MCP Server: Error handling request', {
        error
      });
      try {
        var _transport2;
        await ((_transport2 = transport) === null || _transport2 === void 0 ? void 0 : _transport2.close());
      } catch (closeError) {
        logger.error('MCP Server: Error closing transport during error handling', {
          error: closeError
        });
      }
      if (server) {
        try {
          await server.close();
        } catch (closeError) {
          logger.error('MCP Server: Error closing server during error handling', {
            error: closeError
          });
        }
      }
      logger.error('MCP Server: Error handling request', {
        error
      });
      return response.customError({
        statusCode: 500,
        body: {
          message: `Internal server error: ${error}`,
          attributes: {
            code: _types.ErrorCode.InternalError
          }
        }
      });
    }
  }));
  router.versioned.get({
    path: MCP_SERVER_PATH,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'MCP server',
    description: TECHNICAL_PREVIEW_WARNING,
    options: {
      tags: ['mcp'],
      availability: {
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: false
  }, wrapHandler(async (ctx, _, response) => {
    const {
      uiSettings
    } = await ctx.core;
    const enabled = await uiSettings.client.get(_constants.ONECHAT_MCP_SERVER_UI_SETTING_ID);
    if (!enabled) {
      return response.notFound();
    }
    return response.customError({
      statusCode: 405,
      body: {
        message: 'Method not allowed',
        attributes: {
          code: _types.ErrorCode.ConnectionClosed
        }
      }
    });
  }));
  router.versioned.delete({
    path: MCP_SERVER_PATH,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'MCP server',
    description: TECHNICAL_PREVIEW_WARNING,
    options: {
      tags: ['mcp'],
      xsrfRequired: false,
      availability: {
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: false
  }, wrapHandler(async (ctx, _, response) => {
    const {
      uiSettings
    } = await ctx.core;
    const enabled = await uiSettings.client.get(_constants.ONECHAT_MCP_SERVER_UI_SETTING_ID);
    if (!enabled) {
      return response.notFound();
    }
    return response.customError({
      statusCode: 405,
      body: {
        message: 'Method not allowed',
        attributes: {
          code: _types.ErrorCode.ConnectionClosed
        }
      }
    });
  }));
}