"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatRoutes = registerChatRoutes;
var _configSchema = require("@kbn/config-schema");
var _rxjs = require("rxjs");
var _sseUtilsServer = require("@kbn/sse-utils-server");
var _onechatCommon = require("@kbn/onechat-common");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerChatRoutes({
  router,
  getInternalServices,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });
  router.post({
    path: '/internal/onechat/chat',
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    validate: {
      query: _configSchema.schema.object({
        stream: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      body: _configSchema.schema.object({
        agentId: _configSchema.schema.string({
          defaultValue: _onechatCommon.OneChatDefaultAgentId
        }),
        mode: _configSchema.schema.oneOf([_configSchema.schema.literal(_onechatCommon.AgentMode.normal), _configSchema.schema.literal(_onechatCommon.AgentMode.reason), _configSchema.schema.literal(_onechatCommon.AgentMode.plan), _configSchema.schema.literal(_onechatCommon.AgentMode.research)], {
          defaultValue: _onechatCommon.AgentMode.normal
        }),
        connectorId: _configSchema.schema.maybe(_configSchema.schema.string()),
        conversationId: _configSchema.schema.maybe(_configSchema.schema.string()),
        nextMessage: _configSchema.schema.string()
      })
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      chat: chatService
    } = getInternalServices();
    const {
      agentId,
      mode,
      connectorId,
      conversationId,
      nextMessage
    } = request.body;
    const {
      stream
    } = request.query;
    const abortController = new AbortController();
    request.events.aborted$.subscribe(() => {
      abortController.abort();
    });
    const chatEvents$ = chatService.converse({
      agentId,
      mode,
      connectorId,
      conversationId,
      nextInput: {
        message: nextMessage
      },
      request
    });
    if (stream) {
      return response.ok({
        body: (0, _sseUtilsServer.observableIntoEventSourceStream)(chatEvents$, {
          signal: abortController.signal,
          logger
        })
      });
    } else {
      const events = await (0, _rxjs.firstValueFrom)(chatEvents$.pipe((0, _rxjs.toArray)()));
      const {
        data: {
          round
        }
      } = events.find(_onechatCommon.isRoundCompleteEvent);
      const {
        data: {
          conversationId: convId
        }
      } = events.find(e => (0, _onechatCommon.isConversationUpdatedEvent)(e) || (0, _onechatCommon.isConversationCreatedEvent)(e));
      return response.ok({
        body: {
          conversationId: convId,
          steps: round.steps,
          response: round.assistantResponse
        }
      });
    }
  }));
}