"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flattenMappings = exports.cleanupMapping = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a flattened representation of the mappings, with all fields at the top level.
 */
const flattenMappings = ({
  mappings
}) => {
  var _mappings$properties;
  const properties = (_mappings$properties = mappings.properties) !== null && _mappings$properties !== void 0 ? _mappings$properties : {};
  function extractFields(obj, prefix = '') {
    let fields = [];
    for (const [key, value] of Object.entries(obj)) {
      const fieldPath = prefix ? `${prefix}.${key}` : key;
      if (value.type) {
        // If it's a leaf field, add it
        fields.push({
          type: value.type,
          path: fieldPath
        });
      } else if (value.properties) {
        // If it's an object, go deeper
        fields = fields.concat(extractFields(value.properties, fieldPath));
      }
    }
    return fields;
  }
  return extractFields(properties);
};

/**
 * Remove non-relevant mapping information such as `ignore_above` to reduce overall token length of response
 * @param mapping
 */
exports.flattenMappings = flattenMappings;
const cleanupMapping = mapping => {
  const recurseKeys = ['properties', 'fields'];
  const fieldsToKeep = ['type', 'dynamic', '_meta', 'enabled'];
  function recursiveCleanup(obj) {
    if (Array.isArray(obj)) {
      return obj.map(item => recursiveCleanup(item));
    } else if (obj !== null && typeof obj === 'object') {
      const cleaned = {};
      for (const key of Object.keys(obj)) {
        if (recurseKeys.includes(key)) {
          const value = obj[key];
          if (value !== null && typeof value === 'object') {
            // For properties/fields: preserve all keys inside
            const subCleaned = {};
            for (const fieldName of Object.keys(value)) {
              subCleaned[fieldName] = recursiveCleanup(value[fieldName]);
            }
            cleaned[key] = subCleaned;
          }
        } else if (fieldsToKeep.includes(key)) {
          cleaned[key] = recursiveCleanup(obj[key]);
        }
      }
      return cleaned;
    } else {
      return obj;
    }
  }
  return recursiveCleanup(mapping);
};
exports.cleanupMapping = cleanupMapping;