"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteInferenceEndpoint = deleteInferenceEndpoint;
exports.getInferenceEndpointsForEmbedding = void 0;
exports.getKbModelStatus = getKbModelStatus;
exports.isInferenceEndpointMissingOrUnavailable = isInferenceEndpointMissingOrUnavailable;
exports.waitForKbModel = waitForKbModel;
exports.warmupModel = warmupModel;
var _elasticsearch = require("@elastic/elasticsearch");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _common = require("../../common");
var _get_inference_id_from_write_index = require("./knowledge_base_service/get_inference_id_from_write_index");
var _reindex_knowledge_base = require("./knowledge_base_service/reindex_knowledge_base");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUPPORTED_TASK_TYPES = ['sparse_embedding', 'text_embedding'];
const getInferenceEndpointsForEmbedding = async ({
  esClient,
  logger
}) => {
  const {
    endpoints
  } = await esClient.asInternalUser.inference.get({
    inference_id: '_all'
  });
  if (!endpoints.length) {
    throw new Error('Did not find any inference endpoints');
  }
  const embeddingEndpoints = endpoints.filter(endpoint => SUPPORTED_TASK_TYPES.includes(endpoint.task_type));
  if (!embeddingEndpoints.length) {
    throw new Error('Did not find any inference endpoints for embedding');
  }
  logger.debug(`Found ${embeddingEndpoints.length} inference endpoints for supported task types`);
  return {
    inferenceEndpoints: embeddingEndpoints
  };
};
exports.getInferenceEndpointsForEmbedding = getInferenceEndpointsForEmbedding;
async function getInferenceEndpoint({
  esClient,
  inferenceId
}) {
  const response = await esClient.asInternalUser.inference.get({
    inference_id: inferenceId
  });
  if (response.endpoints.length === 0) {
    throw new Error('Inference endpoint not found');
  }
  return response.endpoints[0];
}
async function deleteInferenceEndpoint({
  esClient,
  logger,
  inferenceId
}) {
  try {
    logger.info(`Attempting to delete inference endpoint with ID: ${inferenceId}`);
    await esClient.asInternalUser.inference.delete({
      inference_id: inferenceId
    });
    logger.info(`Successfully deleted inference endpoint with ID: ${inferenceId}`);
  } catch (error) {
    var _error$body, _error$body$error;
    if (error instanceof _elasticsearch.errors.ResponseError && ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : (_error$body$error = _error$body.error) === null || _error$body$error === void 0 ? void 0 : _error$body$error.type) === 'resource_not_found_exception') {
      logger.debug(`Inference endpoint "${inferenceId}" was already deleted. Skipping deletion.`);
      return;
    }
    logger.error(`Failed to delete inference endpoint with ID: ${inferenceId}. Error: ${error.message}`);
  }
}
function isInferenceEndpointMissingOrUnavailable(error) {
  var _error$body2, _error$body2$error, _error$body3, _error$body3$error;
  return error instanceof _elasticsearch.errors.ResponseError && (((_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : (_error$body2$error = _error$body2.error) === null || _error$body2$error === void 0 ? void 0 : _error$body2$error.type) === 'resource_not_found_exception' || ((_error$body3 = error.body) === null || _error$body3 === void 0 ? void 0 : (_error$body3$error = _error$body3.error) === null || _error$body3$error === void 0 ? void 0 : _error$body3$error.type) === 'status_exception');
}
async function getKbModelStatus({
  core,
  esClient,
  logger,
  config,
  inferenceId
}) {
  var _endpoint2, _endpoint2$service_se, _trainedModelStatsRes, _modelStats$deploymen, _modelStats$deploymen2, _modelStats$deploymen3, _modelStats$deploymen4, _modelStats$deploymen5, _modelStats$deploymen6, _modelStats$deploymen7, _modelStats$deploymen8, _modelStats$deploymen9, _modelStats$deploymen10, _modelStats$deploymen11, _modelStats$deploymen12, _modelStats$deploymen13, _modelStats$deploymen14;
  const enabled = config.enableKnowledgeBase;
  const concreteWriteIndex = await (0, _get_inference_id_from_write_index.getConcreteWriteIndex)(esClient, logger);
  const isReIndexing = await (0, _reindex_knowledge_base.isReIndexInProgress)({
    esClient,
    logger,
    core
  });
  const currentInferenceId = await (0, _get_inference_id_from_write_index.getInferenceIdFromWriteIndex)(esClient, logger);
  if (!inferenceId) {
    if (!currentInferenceId) {
      return {
        enabled,
        errorMessage: 'Inference ID not found in write index',
        currentInferenceId: undefined,
        kbState: _common.KnowledgeBaseState.NOT_INSTALLED,
        concreteWriteIndex,
        isReIndexing
      };
    }
    inferenceId = currentInferenceId;
  }

  // check if inference ID is an EIS inference ID
  const isPreConfiguredInferenceIdInEIS = _common.EIS_PRECONFIGURED_INFERENCE_IDS.includes(inferenceId);
  let endpoint;
  try {
    var _endpoint, _endpoint$service_set;
    endpoint = await getInferenceEndpoint({
      esClient,
      inferenceId
    });
    logger.debug(`Inference endpoint "${inferenceId}" found with model id "${(_endpoint = endpoint) === null || _endpoint === void 0 ? void 0 : (_endpoint$service_set = _endpoint.service_settings) === null || _endpoint$service_set === void 0 ? void 0 : _endpoint$service_set.model_id}"`);

    // if the endpoint is in EIS, the model doesn't have to be downloaded and deployed
    // Therefore, return the KB state as READY if the endpoint exists
    if (isPreConfiguredInferenceIdInEIS && endpoint.service === 'elastic') {
      return {
        endpoint,
        enabled,
        kbState: _common.KnowledgeBaseState.READY,
        currentInferenceId,
        concreteWriteIndex,
        isReIndexing
      };
    }
  } catch (error) {
    if (!isInferenceEndpointMissingOrUnavailable(error)) {
      throw error;
    }
    logger.warn(`Inference endpoint "${inferenceId}" not found or unavailable: ${error.message}`);
    return {
      enabled,
      errorMessage: error.message,
      kbState: _common.KnowledgeBaseState.NOT_INSTALLED,
      currentInferenceId,
      concreteWriteIndex,
      isReIndexing
    };
  }
  const modelId = (_endpoint2 = endpoint) === null || _endpoint2 === void 0 ? void 0 : (_endpoint2$service_se = _endpoint2.service_settings) === null || _endpoint2$service_se === void 0 ? void 0 : _endpoint2$service_se.model_id;
  let trainedModelStatsResponse;
  try {
    trainedModelStatsResponse = await esClient.asInternalUser.ml.getTrainedModelsStats({
      model_id: modelId
    });
  } catch (error) {
    logger.debug(`Failed to get model stats for model "${modelId}" and inference id ${inferenceId}: ${error.message}`);
    return {
      enabled,
      endpoint,
      errorMessage: error.message,
      kbState: _common.KnowledgeBaseState.NOT_INSTALLED,
      currentInferenceId,
      concreteWriteIndex,
      isReIndexing
    };
  }
  const modelStats = trainedModelStatsResponse.trained_model_stats.find(stats => {
    var _stats$deployment_sta;
    return ((_stats$deployment_sta = stats.deployment_stats) === null || _stats$deployment_sta === void 0 ? void 0 : _stats$deployment_sta.deployment_id) === inferenceId;
  });
  let kbState;
  if ((_trainedModelStatsRes = trainedModelStatsResponse.trained_model_stats) !== null && _trainedModelStatsRes !== void 0 && _trainedModelStatsRes.length && !modelStats) {
    // model has been deployed at least once, but stopped later
    kbState = _common.KnowledgeBaseState.MODEL_PENDING_DEPLOYMENT;
  } else if ((modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen = modelStats.deployment_stats) === null || _modelStats$deploymen === void 0 ? void 0 : _modelStats$deploymen.state) === 'failed') {
    kbState = _common.KnowledgeBaseState.ERROR;
  } else if ((modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen2 = modelStats.deployment_stats) === null || _modelStats$deploymen2 === void 0 ? void 0 : _modelStats$deploymen2.state) === 'starting' && (modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen3 = modelStats.deployment_stats) === null || _modelStats$deploymen3 === void 0 ? void 0 : (_modelStats$deploymen4 = _modelStats$deploymen3.allocation_status) === null || _modelStats$deploymen4 === void 0 ? void 0 : _modelStats$deploymen4.allocation_count) === 0) {
    kbState = _common.KnowledgeBaseState.DEPLOYING_MODEL;
  } else if ((modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen5 = modelStats.deployment_stats) === null || _modelStats$deploymen5 === void 0 ? void 0 : _modelStats$deploymen5.state) === 'started' && (modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen6 = modelStats.deployment_stats) === null || _modelStats$deploymen6 === void 0 ? void 0 : (_modelStats$deploymen7 = _modelStats$deploymen6.allocation_status) === null || _modelStats$deploymen7 === void 0 ? void 0 : _modelStats$deploymen7.state) === 'fully_allocated' && (modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen8 = modelStats.deployment_stats) === null || _modelStats$deploymen8 === void 0 ? void 0 : (_modelStats$deploymen9 = _modelStats$deploymen8.allocation_status) === null || _modelStats$deploymen9 === void 0 ? void 0 : _modelStats$deploymen9.allocation_count) > 0) {
    kbState = _common.KnowledgeBaseState.READY;
  } else if ((modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen10 = modelStats.deployment_stats) === null || _modelStats$deploymen10 === void 0 ? void 0 : _modelStats$deploymen10.state) === 'started' && (modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen11 = modelStats.deployment_stats) === null || _modelStats$deploymen11 === void 0 ? void 0 : (_modelStats$deploymen12 = _modelStats$deploymen11.allocation_status) === null || _modelStats$deploymen12 === void 0 ? void 0 : _modelStats$deploymen12.state) === 'fully_allocated' && (modelStats === null || modelStats === void 0 ? void 0 : (_modelStats$deploymen13 = modelStats.deployment_stats) === null || _modelStats$deploymen13 === void 0 ? void 0 : (_modelStats$deploymen14 = _modelStats$deploymen13.allocation_status) === null || _modelStats$deploymen14 === void 0 ? void 0 : _modelStats$deploymen14.allocation_count) === 0) {
    // model has been scaled down due to inactivity
    kbState = _common.KnowledgeBaseState.MODEL_PENDING_ALLOCATION;
  } else {
    kbState = _common.KnowledgeBaseState.ERROR;
  }
  return {
    endpoint,
    enabled,
    modelStats,
    kbState,
    concreteWriteIndex,
    currentInferenceId,
    isReIndexing
  };
}
async function waitForKbModel({
  core,
  esClient,
  logger,
  config,
  inferenceId
}) {
  logger.debug(`Waiting for knowledge base model to be ready for inference ID "${inferenceId}" !!`);

  // Run a dummy inference to trigger the model to deploy
  // This is a workaround for the fact that the model may not be deployed yet
  await warmupModel({
    esClient,
    logger,
    inferenceId
  }).catch(() => {});
  return (0, _pRetry.default)(async () => {
    logger.debug(`Checking knowledge base model status for inference ID "${inferenceId}"`);
    const {
      kbState
    } = await getKbModelStatus({
      core,
      esClient,
      logger,
      config,
      inferenceId
    });
    if (kbState !== _common.KnowledgeBaseState.READY) {
      const message = `Knowledge base model is not yet ready. kbState = ${kbState}, `;
      logger.debug(message);
      throw new Error(message);
    }
    logger.debug('Knowledge base model is ready.');
  }, {
    retries: 30,
    factor: 2,
    maxTimeout: 30_000
  });
}
async function warmupModel({
  esClient,
  logger,
  inferenceId
}) {
  logger.debug(`Warming up model for "${inferenceId}"`);
  await (0, _pRetry.default)(() => esClient.asInternalUser.inference.inference({
    inference_id: inferenceId,
    input: 'hello world'
  }), {
    retries: 10
  }).catch(error => {
    logger.error(`Unable to warm up model for "${inferenceId}": ${error.message}`);
  });
}