"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildCustomUrlFromSettings = buildCustomUrlFromSettings;
exports.getNewCustomUrlDefaults = getNewCustomUrlDefaults;
exports.getQueryEntityFieldNames = getQueryEntityFieldNames;
exports.getSupportedFieldNames = getSupportedFieldNames;
exports.getTestUrl = getTestUrl;
exports.isValidCustomUrlSettings = isValidCustomUrlSettings;
exports.isValidCustomUrlSettingsTimeRange = isValidCustomUrlSettingsTimeRange;
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _rison = _interopRequireDefault(require("@kbn/rison"));
var _url = _interopRequireDefault(require("url"));
var _public = require("@kbn/kibana-utils-plugin/public");
var _public2 = require("@kbn/dashboard-plugin/public");
var _esQuery = require("@kbn/es-query");
var _mlDataFrameAnalyticsUtils = require("@kbn/ml-data-frame-analytics-utils");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _mlParseInterval = require("@kbn/ml-parse-interval");
var _deeplinksAnalytics = require("@kbn/deeplinks-analytics");
var _fields_utils = require("../../../../../common/util/fields_utils");
var _constants = require("./constants");
var _job_utils = require("../../../../../common/util/job_utils");
var _string_utils = require("../../../util/string_utils");
var _custom_url_utils = require("../../../util/custom_url_utils");
var _anomaly_detection_jobs = require("../../../../../common/types/anomaly_detection_jobs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getNewCustomUrlDefaults(job, dashboards, dataViews, isPartialDFAJob) {
  var _dataViews$find, _ref;
  // Returns the settings object in the format used by the custom URL editor
  // for a new custom URL.
  const kibanaSettings = {
    queryFieldNames: []
  };

  // Set the default type.
  let urlType = _constants.URL_TYPE.OTHER;
  if (dashboards !== undefined && dashboards.length > 0) {
    urlType = _constants.URL_TYPE.KIBANA_DASHBOARD;
    kibanaSettings.dashboardId = dashboards[0].id;
  } else if (dataViews !== undefined && dataViews.length > 0) {
    urlType = _constants.URL_TYPE.KIBANA_DISCOVER;
  }

  // For the Discover option, set the default data view to that
  // which matches the indices configured in the job datafeed.
  let query = {};
  let indicesName;
  let backupIndicesName;
  let backupDataViewId;
  let jobId;
  if ((0, _anomaly_detection_jobs.isAnomalyDetectionJob)(job) && dataViews !== undefined && dataViews.length > 0 && job.datafeed_config !== undefined && job.datafeed_config.indices !== undefined && job.datafeed_config.indices.length > 0) {
    var _job$datafeed_config$, _job$datafeed_config;
    indicesName = job.datafeed_config.indices.join();
    query = (_job$datafeed_config$ = (_job$datafeed_config = job.datafeed_config) === null || _job$datafeed_config === void 0 ? void 0 : _job$datafeed_config.query) !== null && _job$datafeed_config$ !== void 0 ? _job$datafeed_config$ : {};
    jobId = job.job_id;
  } else if (((0, _mlDataFrameAnalyticsUtils.isDataFrameAnalyticsConfigs)(job) || isPartialDFAJob) && dataViews !== undefined && dataViews.length > 0) {
    var _dfaJob$source$query, _dfaJob$source;
    // Ensure cast as dfaJob if it's just a partial from the wizard
    const dfaJob = job;
    const sourceIndex = Array.isArray(dfaJob.source.index) ? dfaJob.source.index.join() : dfaJob.source.index;
    indicesName = isPartialDFAJob ? sourceIndex : dfaJob.dest.index;
    backupIndicesName = sourceIndex;
    query = (_dfaJob$source$query = (_dfaJob$source = dfaJob.source) === null || _dfaJob$source === void 0 ? void 0 : _dfaJob$source.query) !== null && _dfaJob$source$query !== void 0 ? _dfaJob$source$query : {};
    jobId = dfaJob.id;
  }
  const defaultDataViewId = (_dataViews$find = dataViews.find(dv => dv.title === indicesName)) === null || _dataViews$find === void 0 ? void 0 : _dataViews$find.id;
  if (defaultDataViewId === undefined && backupIndicesName !== undefined) {
    var _dataViews$find2;
    backupDataViewId = (_dataViews$find2 = dataViews.find(dv => dv.title === backupIndicesName)) === null || _dataViews$find2 === void 0 ? void 0 : _dataViews$find2.id;
  }
  kibanaSettings.discoverIndexPatternId = (_ref = defaultDataViewId !== null && defaultDataViewId !== void 0 ? defaultDataViewId : backupDataViewId) !== null && _ref !== void 0 ? _ref : '';
  kibanaSettings.filters = defaultDataViewId === null ? [] : (0, _job_utils.getFiltersForDSLQuery)(query, defaultDataViewId, jobId);
  return {
    label: '',
    type: urlType,
    // Note timeRange is only editable in new URLs for Dashboard and Discover URLs,
    // as for other URLs we have no way of knowing how the field will be used in the URL.
    timeRange: {
      type: _constants.TIME_RANGE_TYPE.AUTO,
      interval: ''
    },
    kibanaSettings,
    otherUrlSettings: {
      urlValue: ''
    }
  };
}

// Returns the list of supported field names that can be used
// to add to the query used when linking to a Kibana dashboard or Discover.
function getSupportedFieldNames(job, dataView) {
  var _dataView$fields$getA;
  const sortedFields = (_dataView$fields$getA = dataView.fields.getAll().sort((a, b) => a.name.localeCompare(b.name))) !== null && _dataView$fields$getA !== void 0 ? _dataView$fields$getA : [];
  let filterFunction = field => _fields_utils.categoryFieldTypes.some(type => {
    var _field$esTypes;
    return (_field$esTypes = field.esTypes) === null || _field$esTypes === void 0 ? void 0 : _field$esTypes.includes(type);
  });
  if ((0, _mlDataFrameAnalyticsUtils.isDataFrameAnalyticsConfigs)(job)) {
    const resultsField = job.dest.results_field;
    filterFunction = f => _fields_utils.categoryFieldTypes.some(type => {
      var _f$esTypes;
      return (_f$esTypes = f.esTypes) === null || _f$esTypes === void 0 ? void 0 : _f$esTypes.includes(type);
    }) && !f.name.startsWith(resultsField !== null && resultsField !== void 0 ? resultsField : _mlDataFrameAnalyticsUtils.DEFAULT_RESULTS_FIELD);
  }
  const categoryFields = sortedFields.filter(filterFunction);
  return categoryFields.map(field => field.name);
}
function getQueryEntityFieldNames(job) {
  // Returns the list of partitioning and influencer field names that can be used
  // as entities to add to the query used when linking to a Kibana dashboard or Discover.
  const influencers = job.analysis_config.influencers;
  const detectors = job.analysis_config.detectors;
  const entityFieldNames = [];
  if (influencers !== undefined) {
    entityFieldNames.push(...influencers);
  }
  detectors.forEach((detector, detectorIndex) => {
    const partitioningFields = (0, _job_utils.getPartitioningFieldNames)(job, detectorIndex);
    partitioningFields.forEach(fieldName => {
      if (entityFieldNames.indexOf(fieldName) === -1) {
        entityFieldNames.push(fieldName);
      }
    });
  });
  return entityFieldNames;
}
function isValidCustomUrlSettingsTimeRange(timeRangeSettings) {
  if (timeRangeSettings.type === _constants.TIME_RANGE_TYPE.INTERVAL) {
    const interval = (0, _mlParseInterval.parseInterval)(timeRangeSettings.interval);
    return interval !== null;
  }
  return true;
}
function isValidCustomUrlSettings(settings, savedCustomUrls) {
  let isValid = (0, _custom_url_utils.isValidLabel)(settings.label, savedCustomUrls);
  if (isValid === true) {
    isValid = isValidCustomUrlSettingsTimeRange(settings.timeRange);
  }
  return isValid;
}
function buildCustomUrlFromSettings(dashboardService, share, settings) {
  // Dashboard URL returns a Promise as a query is made to obtain the full dashboard config.
  // So wrap the other two return types in a Promise for consistent return type.
  if (settings.type === _constants.URL_TYPE.KIBANA_DASHBOARD) {
    return buildDashboardUrlFromSettings(dashboardService, share, settings);
  } else if (settings.type === _constants.URL_TYPE.KIBANA_DISCOVER) {
    return Promise.resolve(buildDiscoverUrlFromSettings(settings));
  } else {
    var _settings$otherUrlSet, _settings$otherUrlSet2;
    const urlToAdd = {
      url_name: settings.label,
      url_value: (_settings$otherUrlSet = (_settings$otherUrlSet2 = settings.otherUrlSettings) === null || _settings$otherUrlSet2 === void 0 ? void 0 : _settings$otherUrlSet2.urlValue) !== null && _settings$otherUrlSet !== void 0 ? _settings$otherUrlSet : '',
      ...(settings.customTimeRange ? {
        is_custom_time_range: true
      } : {})
    };
    return Promise.resolve(urlToAdd);
  }
}
function getUrlRangeFromSettings(settings) {
  var _customStart, _customEnd;
  let customStart;
  let customEnd;
  if (settings.customTimeRange && settings.customTimeRange.start && settings.customTimeRange.end) {
    customStart = settings.customTimeRange.start.toISOString();
    customEnd = settings.customTimeRange.end.toISOString();
  }
  return {
    from: (_customStart = customStart) !== null && _customStart !== void 0 ? _customStart : '$earliest$',
    to: (_customEnd = customEnd) !== null && _customEnd !== void 0 ? _customEnd : '$latest$'
  };
}
async function buildDashboardUrlFromSettings(dashboardService, share, settings, isPartialDFAJob) {
  var _settings$kibanaSetti, _settings$kibanaSetti2, _share$url$locators$g, _state$filters;
  // Get the complete list of attributes for the selected dashboard (query, filters).
  const {
    dashboardId,
    queryFieldNames
  } = (_settings$kibanaSetti = settings.kibanaSettings) !== null && _settings$kibanaSetti !== void 0 ? _settings$kibanaSetti : {};
  if (!dashboardService) {
    throw Error(`Missing dashboard service (got ${dashboardService})`);
  }
  if (!(0, _mlIsDefined.isDefined)(dashboardId)) {
    throw Error(`DashboardId is invalid (got ${dashboardId})`);
  }
  const findDashboardsService = await dashboardService.findDashboardsService();
  const responses = await findDashboardsService.findByIds([dashboardId]);
  if (!responses || responses.length === 0 || responses[0].status === 'error') {
    throw Error(`Unable to find dashboard with id ${dashboardId} (got ${responses})`);
  }
  const dashboard = responses[0];

  // Query from the datafeed config will be saved as custom filters
  // Use them if there are set.
  let filters = settings === null || settings === void 0 ? void 0 : (_settings$kibanaSetti2 = settings.kibanaSettings) === null || _settings$kibanaSetti2 === void 0 ? void 0 : _settings$kibanaSetti2.filters;

  // Use the query from the dashboard only if no job entities are selected.
  let query;

  // Override with filters and queries from saved dashboard if they are available.
  const {
    searchSource
  } = dashboard.attributes.kibanaSavedObjectMeta;
  if (searchSource !== undefined) {
    if (Array.isArray(searchSource.filter) && searchSource.filter.length > 0) {
      filters = searchSource.filter;
    }
    query = searchSource.query;
  }
  const queryFromEntityFieldNames = buildAppStateQueryParam(queryFieldNames !== null && queryFieldNames !== void 0 ? queryFieldNames : []);
  if (queryFromEntityFieldNames !== undefined) {
    query = queryFromEntityFieldNames;
  }
  const {
    from,
    to
  } = getUrlRangeFromSettings(settings);
  const location = await ((_share$url$locators$g = share.url.locators.get(_deeplinksAnalytics.DASHBOARD_APP_LOCATOR)) === null || _share$url$locators$g === void 0 ? void 0 : _share$url$locators$g.getLocation({
    dashboardId,
    timeRange: {
      from,
      to,
      mode: 'absolute'
    },
    filters,
    query,
    // Don't hash the URL since this string will be 1. shown to the user and 2. used as a
    // template to inject the time parameters.
    useHash: false
  }));

  // Temp workaround
  const state = location === null || location === void 0 ? void 0 : location.state;
  const resultPath = (0, _public.setStateToKbnUrl)('_a', (0, _public2.cleanEmptyKeys)({
    query: state.query,
    filters: (_state$filters = state.filters) === null || _state$filters === void 0 ? void 0 : _state$filters.filter(f => !(0, _esQuery.isFilterPinned)(f)),
    savedQuery: state.savedQuery
  }), {
    useHash: false,
    storeInHashQuery: true
  }, location === null || location === void 0 ? void 0 : location.path);
  const urlToAdd = {
    url_name: settings.label,
    url_value: decodeURIComponent(`dashboards${_url.default.parse(resultPath).hash}`),
    time_range: _constants.TIME_RANGE_TYPE.AUTO
  };
  if (settings.timeRange.type === _constants.TIME_RANGE_TYPE.INTERVAL) {
    urlToAdd.time_range = settings.timeRange.interval;
  }
  if (settings.customTimeRange) {
    urlToAdd.is_custom_time_range = true;
  }
  return urlToAdd;
}
function buildDiscoverUrlFromSettings(settings) {
  var _settings$kibanaSetti3;
  const {
    discoverIndexPatternId,
    queryFieldNames,
    filters
  } = (_settings$kibanaSetti3 = settings.kibanaSettings) !== null && _settings$kibanaSetti3 !== void 0 ? _settings$kibanaSetti3 : {};

  // Add time settings to the global state URL parameter with $earliest$ and
  // $latest$ tokens which get substituted for times around the time of the
  // anomaly on which the URL will be run against.
  const {
    from,
    to
  } = getUrlRangeFromSettings(settings);
  const _g = _rison.default.encode({
    time: {
      from,
      to,
      mode: 'absolute'
    }
  });

  // Add the index pattern and query to the appState part of the URL.
  const appState = {
    index: discoverIndexPatternId,
    filters
  };
  // If partitioning field entities have been configured add tokens
  // to the URL to use in the Discover page search.

  // Ideally we would put entities in the filters section, but currently this involves creating parameters of the form
  // filters:!(('$state':(store:appState),meta:(alias:!n,disabled:!f,index:b30fd340-efb4-11e7-a600-0f58b1422b87,
  // key:airline,negate:!f,params:(query:AAL,type:phrase),type:phrase,value:AAL),query:(match:(airline:(query:AAL,type:phrase)))))
  // which includes the ID of the index holding the field used in the filter.

  // So for simplicity, put entities in the query, replacing any query which is there already.
  // e.g. query:(language:kuery,query:'region:us-east-1%20and%20instance:i-20d061fa')
  const queryFromEntityFieldNames = buildAppStateQueryParam(queryFieldNames !== null && queryFieldNames !== void 0 ? queryFieldNames : []);
  if (queryFromEntityFieldNames !== undefined) {
    appState.query = queryFromEntityFieldNames;
  }
  const _a = _rison.default.encode(appState);
  const urlValue = `discover#/?_g=${_g}&_a=${_a}`;
  const urlToAdd = {
    url_name: settings.label,
    url_value: urlValue,
    time_range: _constants.TIME_RANGE_TYPE.AUTO
  };
  if (settings.timeRange.type === _constants.TIME_RANGE_TYPE.INTERVAL) {
    urlToAdd.time_range = settings.timeRange.interval;
  }
  if (settings.customTimeRange) {
    urlToAdd.is_custom_time_range = true;
  }
  return urlToAdd;
}

// Builds the query parameter for use in the _a AppState part of a Kibana Dashboard or Discover URL.
function buildAppStateQueryParam(queryFieldNames) {
  let queryParam;
  if (queryFieldNames !== undefined && queryFieldNames.length > 0) {
    let queryString = '';
    queryFieldNames.forEach((fieldName, i) => {
      if (i > 0) {
        queryString += ' and ';
      }
      queryString += `${(0, _string_utils.escapeForElasticsearchQuery)(fieldName)}:"$${fieldName}$"`;
    });
    queryParam = {
      language: 'kuery',
      query: queryString
    };
  }
  return queryParam;
}

// Builds the full URL for testing out a custom URL configuration, which
// may contain dollar delimited partition / influencer entity tokens and
// drilldown time range settings.
async function getAnomalyDetectionJobTestUrl(mlApi, job, customUrl) {
  const interval = (0, _mlParseInterval.parseInterval)(job.analysis_config.bucket_span);
  const bucketSpanSecs = interval !== null ? interval.asSeconds() : 0;

  // By default, return configured url_value. Look to substitute any dollar-delimited
  // tokens with values from the highest scoring anomaly, or if no anomalies, with
  // values from a document returned by the search in the job datafeed.
  let testUrl = customUrl.url_value;

  // Query to look for the highest scoring anomaly.
  const body = {
    query: {
      bool: {
        must: [{
          term: {
            job_id: job.job_id
          }
        }, {
          term: {
            result_type: 'record'
          }
        }]
      }
    },
    size: 1,
    _source: {
      excludes: []
    },
    sort: [{
      record_score: {
        order: 'desc'
      }
    }]
  };
  let resp;
  try {
    resp = await mlApi.results.anomalySearch(body, [job.job_id]);
  } catch (error) {
    // search may fail if the job doesn't already exist
    // ignore this error as the outer function call will raise a toast
  }
  if (resp && resp.hits.total.value > 0) {
    const record = resp.hits.hits[0]._source;
    testUrl = (0, _custom_url_utils.replaceTokensInUrlValue)(customUrl, bucketSpanSecs, record, 'timestamp');
    return testUrl;
  } else {
    // No anomalies yet for this job, so do a preview of the search
    // configured in the job datafeed to obtain sample docs.

    let jobConfig = (0, _lodash.cloneDeep)(job);
    let {
      datafeed_config: datafeedConfig
    } = jobConfig;
    try {
      // attempt load the non-combined job and datafeed so they can be used in the datafeed preview
      // use jobForCloning to omit fields that shouldn't be included in datafeed preview requests
      const response = await mlApi.jobs.jobForCloning(job.job_id);
      if (response !== null && response !== void 0 && response.job && response !== null && response !== void 0 && response.datafeed) {
        datafeedConfig = response.datafeed;
        jobConfig = response.job;
      }
    } catch (error) {
      // jobs may not exist as this might be called from the AD job wizards
      // ignore this error as the outer function call will raise a toast
    }
    if (jobConfig === undefined || datafeedConfig === undefined) {
      return testUrl;
    }
    if (datafeedConfig.authorization !== undefined) {
      delete datafeedConfig.authorization;
    }
    if (datafeedConfig && jobConfig.datafeed_config !== undefined) {
      delete jobConfig.datafeed_config;
    }
    const preview = await mlApi.jobs.datafeedPreview(undefined, jobConfig, datafeedConfig
    // @ts-expect-error TODO: fix after elasticsearch-js bump
    );
    const docTimeFieldName = job.data_description.time_field;

    // Create a dummy object which contains the fields necessary to build the URL.
    const firstBucket = preview[0];
    if (firstBucket !== undefined) {
      testUrl = (0, _custom_url_utils.replaceTokensInUrlValue)(customUrl, bucketSpanSecs, firstBucket, docTimeFieldName);
    }
    return testUrl;
  }
}
async function getDataFrameAnalyticsTestUrl(mlApi, job, customUrl, timeFieldName, currentTimeFilter, isPartialDFAJob) {
  // By default, return configured url_value. Look to substitute any dollar-delimited
  // tokens with values from a sample doc in the destination index
  const sourceIndex = Array.isArray(job.source.index) ? job.source.index.join() : job.source.index;
  let testUrl = customUrl.url_value;
  let record;
  let resp;
  try {
    const body = {
      // Use source index for partial job as there is no dest index yet
      index: isPartialDFAJob ? sourceIndex : job.dest.index,
      body: {
        size: 1
      }
    };
    resp = await mlApi.esSearch(body);
    if (resp && resp.hits.total.value > 0) {
      record = resp.hits.hits[0]._source;
    } else {
      var _resp, _resp$hits;
      // No results for this job yet so use source index for example doc.
      resp = await mlApi.esSearch({
        index: Array.isArray(job.source.index) ? job.source.index.join(',') : job.source.index,
        body: {
          size: 1
        }
      });
      record = (_resp = resp) === null || _resp === void 0 ? void 0 : (_resp$hits = _resp.hits) === null || _resp$hits === void 0 ? void 0 : _resp$hits.hits[0]._source;
    }
    if (record) {
      const timeRangeInterval = customUrl.time_range !== undefined ? (0, _mlParseInterval.parseInterval)(customUrl.time_range) : null;
      if (timeRangeInterval !== null && timeFieldName !== null) {
        const timestamp = record[timeFieldName];
        const configuredUrlValue = customUrl.url_value;
        if (configuredUrlValue.includes('$earliest$')) {
          const earliestMoment = (0, _moment.default)(timestamp);
          earliestMoment.subtract(timeRangeInterval);
          record.earliest = earliestMoment.toISOString(); // e.g. 2016-02-08T16:00:00.000Z
        }
        if (configuredUrlValue.includes('$latest$')) {
          const latestMoment = (0, _moment.default)(timestamp);
          latestMoment.add(timeRangeInterval);
          record.latest = latestMoment.toISOString();
        }
        testUrl = (0, _string_utils.replaceStringTokens)(customUrl.url_value, record, true);
      } else {
        testUrl = (0, _custom_url_utils.replaceTokensInDFAUrlValue)(customUrl, record, currentTimeFilter);
      }
    }
    return testUrl;
  } catch (error) {
    // search may fail if the job doesn't already exist
    // ignore this error as the outer function call will raise a toast
  }
  return testUrl;
}
function getTestUrl(mlApi, job, customUrl, timeFieldName, currentTimeFilter, isPartialDFAJob) {
  if ((0, _mlDataFrameAnalyticsUtils.isDataFrameAnalyticsConfigs)(job) || isPartialDFAJob) {
    return getDataFrameAnalyticsTestUrl(mlApi, job, customUrl, timeFieldName, currentTimeFilter, isPartialDFAJob);
  }
  return getAnomalyDetectionJobTestUrl(mlApi, job, customUrl);
}