"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.writeAttackDiscoveryEvent = void 0;
var _constants = require("../../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_LENGTH = 1024;
const writeAttackDiscoveryEvent = async ({
  action,
  alertsContextCount,
  attackDiscoveryAlertsEnabled,
  authenticatedUser,
  connectorId,
  dataClient,
  duration,
  end,
  eventLogger,
  eventLogIndex,
  executionUuid,
  loadingMessage,
  message,
  newAlerts,
  outcome,
  reason,
  spaceId,
  start
}) => {
  if (attackDiscoveryAlertsEnabled) {
    const alertsCountActive = alertsContextCount != null ? {
      active: alertsContextCount
    } : undefined;
    const alertsCountsNew = newAlerts != null ? {
      new: newAlerts
    } : undefined;
    const metrics = alertsCountActive != null || alertsCountsNew != null ? {
      alert_counts: {
        ...alertsCountActive,
        ...alertsCountsNew
      }
    } : undefined;
    const status = loadingMessage;

    // required because reason is mapped with "ignore_above": 1024, so it won't be returned in the search result if it exceeds this length:
    const trimmedReason = reason != null && reason.length > MAX_LENGTH ? reason.substring(0, MAX_LENGTH) : reason;
    const attackDiscoveryEvent = {
      '@timestamp': new Date().toISOString(),
      event: {
        action,
        // e.g. generation-started, generation-succeeded, generation-failed
        dataset: connectorId,
        // The connector id for this generation
        duration,
        // The duration of a successful generation in nanoseconds
        end: end === null || end === void 0 ? void 0 : end.toISOString(),
        // When generation ended
        outcome,
        // The outcome of the generation (success or failure)
        provider: _constants.ATTACK_DISCOVERY_EVENT_PROVIDER,
        // The plugin-registered provider name
        reason: trimmedReason,
        // for failed generations
        start: start === null || start === void 0 ? void 0 : start.toISOString() // When generation started
      },
      kibana: {
        alert: {
          rule: {
            consumer: 'siem',
            execution: {
              metrics,
              status,
              uuid: executionUuid // The unique identifier for the generation
            }
          }
        },
        space_ids: [spaceId] // The Kibana space ID
      },
      message,
      tags: ['securitySolution', 'attackDiscovery'],
      user: {
        name: authenticatedUser.username // only user.name is supported
      }
    };
    try {
      eventLogger.logEvent(attackDiscoveryEvent);
    } finally {
      await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.refreshEventLogIndex(eventLogIndex));
    }
  }
};
exports.writeAttackDiscoveryEvent = writeAttackDiscoveryEvent;