"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLatestVulnerabilitiesGrouping = exports.isVulnerabilitiesRootGroupingAggregation = void 0;
var _grouping = require("@kbn/grouping");
var _src = require("@kbn/grouping/src");
var _react = require("react");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _esQuery = require("@kbn/es-query");
var _helpers = require("@kbn/grouping/src/containers/query/helpers");
var _constants = require("../../../common/constants");
var _data_view_context = require("../../../common/contexts/data_view_context");
var _use_grouped_vulnerabilities = require("./use_grouped_vulnerabilities");
var _constants2 = require("../constants");
var _cloud_security_grouping = require("../../../components/cloud_security_grouping");
var _translations = require("../translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTermAggregation = (key, field) => ({
  [key]: {
    terms: {
      field,
      size: 1
    }
  }
});
const getAggregationsByGroupField = field => {
  if ((0, _src.isNoneGroup)([field])) {
    return [];
  }
  const aggMetrics = [{
    groupByField: {
      cardinality: {
        field
      }
    },
    critical: {
      filter: {
        term: {
          'vulnerability.severity': {
            value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.CRITICAL,
            case_insensitive: true
          }
        }
      }
    },
    high: {
      filter: {
        term: {
          'vulnerability.severity': {
            value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.HIGH,
            case_insensitive: true
          }
        }
      }
    },
    medium: {
      filter: {
        term: {
          'vulnerability.severity': {
            value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.MEDIUM,
            case_insensitive: true
          }
        }
      }
    },
    low: {
      filter: {
        term: {
          'vulnerability.severity': {
            value: _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.LOW,
            case_insensitive: true
          }
        }
      }
    }
  }];
  switch (field) {
    case _constants.VULNERABILITY_GROUPING_OPTIONS.RESOURCE_ID:
      return [...aggMetrics, getTermAggregation('resourceName', _constants.VULNERABILITY_FIELDS.RESOURCE_NAME)];
    case _constants.VULNERABILITY_GROUPING_OPTIONS.CLOUD_ACCOUNT_ID:
      return [...aggMetrics, getTermAggregation('cloudProvider', _constants.VULNERABILITY_FIELDS.CLOUD_PROVIDER)];
    case _constants.VULNERABILITY_GROUPING_OPTIONS.CVE:
      return [...aggMetrics, getTermAggregation('description', _constants.VULNERABILITY_FIELDS.DESCRIPTION)];
  }
  return aggMetrics;
};

/**
 * Get runtime mappings for the given group field
 * Some fields require additional runtime mappings to aggregate additional information
 * Fallback to keyword type to support custom fields grouping
 */
const getRuntimeMappingsByGroupField = field => {
  if (_constants.CDR_VULNERABILITY_GROUPING_RUNTIME_MAPPING_FIELDS !== null && _constants.CDR_VULNERABILITY_GROUPING_RUNTIME_MAPPING_FIELDS !== void 0 && _constants.CDR_VULNERABILITY_GROUPING_RUNTIME_MAPPING_FIELDS[field]) {
    return _constants.CDR_VULNERABILITY_GROUPING_RUNTIME_MAPPING_FIELDS[field].reduce((acc, runtimeField) => ({
      ...acc,
      [runtimeField]: {
        type: 'keyword'
      }
    }), {});
  }
  return {};
};

/**
 * Returns the root aggregations query for the vulnerabilities grouping
 */
const getRootAggregations = currentSelectedGroup => {
  // Skip creating null group if "None" is selected
  if ((0, _src.isNoneGroup)([currentSelectedGroup])) {
    return [{}];
  }
  const shouldFlattenMultiValueField = (0, _helpers.checkIsFlattenResults)(currentSelectedGroup, _constants.VULNERABILITY_GROUPING_MULTIPLE_VALUE_FIELDS);

  // Create null group filter based on whether we need to flatten results
  const nullGroupFilter = shouldFlattenMultiValueField ? {
    // For multi-value fields, check if field doesn't exist OR has too many values
    bool: {
      should: [{
        bool: {
          must_not: {
            exists: {
              field: currentSelectedGroup
            }
          }
        }
      }, {
        script: {
          script: {
            source: `doc['${currentSelectedGroup}'].size() > ${_src.MAX_RUNTIME_FIELD_SIZE}`,
            lang: 'painless'
          }
        }
      }],
      minimum_should_match: 1
    }
  } : undefined; // Not used for simple fields

  return [{
    nullGroupItems: shouldFlattenMultiValueField ? {
      filter: nullGroupFilter
    } : {
      missing: {
        field: currentSelectedGroup
      }
    }
  }];
};

/**
 * Type Guard for checking if the given source is a VulnerabilitiesRootGroupingAggregation
 */
const isVulnerabilitiesRootGroupingAggregation = groupData => {
  var _groupData$unitsCount;
  return (groupData === null || groupData === void 0 ? void 0 : (_groupData$unitsCount = groupData.unitsCount) === null || _groupData$unitsCount === void 0 ? void 0 : _groupData$unitsCount.value) !== undefined;
};

/**
 * Utility hook to get the latest vulnerabilities grouping data
 * for the vulnerabilities page
 */
exports.isVulnerabilitiesRootGroupingAggregation = isVulnerabilitiesRootGroupingAggregation;
const useLatestVulnerabilitiesGrouping = ({
  groupPanelRenderer,
  getGroupStats,
  groupingLevel = 0,
  groupFilters = [],
  selectedGroup
}) => {
  var _groupData$unitsCount2;
  const {
    dataView
  } = (0, _data_view_context.useDataViewContext)();
  const {
    activePageIndex,
    grouping,
    pageSize,
    query,
    onChangeGroupsItemsPerPage,
    onChangeGroupsPage,
    urlQuery,
    setUrlQuery,
    uniqueValue,
    isNoneSelected,
    onResetFilters,
    error,
    filters,
    setActivePageIndex
  } = (0, _cloud_security_grouping.useCloudSecurityGrouping)({
    dataView,
    groupingTitle: _translations.groupingTitle,
    defaultGroupingOptions: _constants2.defaultGroupingOptions,
    getDefaultQuery: _constants2.getDefaultQuery,
    unit: _translations.VULNERABILITIES_UNIT,
    groupPanelRenderer,
    getGroupStats,
    groupingLocalStorageKey: _constants.LOCAL_STORAGE_VULNERABILITIES_GROUPING_KEY,
    groupingLevel,
    groupsUnit: _translations.VULNERABILITIES_GROUPS_UNIT
  });
  const additionalFilters = (0, _esQuery.buildEsQuery)(dataView, [], groupFilters);
  const currentSelectedGroup = selectedGroup || grouping.selectedGroups[0];
  const groupingQuery = (0, _grouping.getGroupingQuery)({
    additionalFilters: query ? [query, additionalFilters] : [additionalFilters],
    groupByField: currentSelectedGroup,
    uniqueValue,
    timeRange: {
      from: `now-${_cloudSecurityPostureCommon.CDR_EXTENDED_VULN_RETENTION_POLICY}`,
      to: 'now'
    },
    pageNumber: activePageIndex * pageSize,
    size: pageSize,
    sort: [{
      groupByField: {
        order: 'desc'
      }
    }],
    statsAggregations: getAggregationsByGroupField(currentSelectedGroup),
    runtimeMappings: getRuntimeMappingsByGroupField(currentSelectedGroup),
    rootAggregations: getRootAggregations(currentSelectedGroup),
    multiValueFieldsToFlatten: _constants.VULNERABILITY_GROUPING_MULTIPLE_VALUE_FIELDS,
    countByKeyForMultiValueFields: _constants.EVENT_ID
  });
  const {
    data,
    isFetching
  } = (0, _use_grouped_vulnerabilities.useGroupedVulnerabilities)({
    query: groupingQuery,
    enabled: !isNoneSelected
  });
  const groupData = (0, _react.useMemo)(() => (0, _src.parseGroupingQuery)(currentSelectedGroup, uniqueValue, data), [data, currentSelectedGroup, uniqueValue]);
  const isEmptyResults = !isFetching && isVulnerabilitiesRootGroupingAggregation(groupData) && ((_groupData$unitsCount2 = groupData.unitsCount) === null || _groupData$unitsCount2 === void 0 ? void 0 : _groupData$unitsCount2.value) === 0;
  return {
    groupData,
    grouping,
    isFetching,
    activePageIndex,
    setActivePageIndex,
    pageSize,
    selectedGroup,
    onChangeGroupsItemsPerPage,
    onChangeGroupsPage,
    urlQuery,
    setUrlQuery,
    isGroupSelected: !isNoneSelected,
    isGroupLoading: !data,
    onResetFilters,
    filters,
    error,
    isEmptyResults
  };
};
exports.useLatestVulnerabilitiesGrouping = useLatestVulnerabilitiesGrouping;