"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTraceDocsPerPage = getTraceDocsPerPage;
var _common = require("@kbn/observability-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _utils2 = require("../span_links/utils");
var _get_trace_items = require("./get_trace_items");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const requiredFields = (0, _as_mutable_array.asMutableArray)([_apm.AGENT_NAME, _apm.TIMESTAMP_US, _apm.TRACE_ID, _apm.SERVICE_NAME, _apm.PROCESSOR_EVENT]);
const requiredTxFields = (0, _as_mutable_array.asMutableArray)([_apm.TRANSACTION_ID, _apm.TRANSACTION_DURATION, _apm.TRANSACTION_NAME, _apm.TRANSACTION_TYPE]);
const requiredSpanFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_TYPE, _apm.SPAN_NAME, _apm.SPAN_DURATION]);
const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.PARENT_ID, _apm.SERVICE_ENVIRONMENT, _apm.EVENT_OUTCOME, _apm.TRANSACTION_RESULT, _apm.FAAS_COLDSTART, _apm.SPAN_SUBTYPE, _apm.SPAN_ACTION, _apm.SPAN_COMPOSITE_COUNT, _apm.SPAN_COMPOSITE_COMPRESSION_STRATEGY, _apm.SPAN_COMPOSITE_SUM, _apm.SPAN_SYNC, _apm.SPAN_DESTINATION_SERVICE_RESOURCE, _apm.CHILD_ID, _apm.OTEL_SPAN_LINKS_SPAN_ID, _apm.OTEL_SPAN_LINKS_TRACE_ID]);
async function getTraceDocsPerPage({
  apmEventClient,
  maxTraceItems,
  traceId,
  start,
  end,
  searchAfter
}) {
  const size = Math.min(maxTraceItems, _get_trace_items.MAX_ITEMS_PER_PAGE);
  const body = {
    track_total_hits: true,
    size,
    search_after: searchAfter,
    _source: [_apm.SPAN_LINKS],
    query: {
      bool: {
        filter: [{
          term: {
            [_apm.TRACE_ID]: traceId
          }
        }, ...(0, _server.rangeQuery)(start, end)],
        should: {
          exists: {
            field: _apm.PARENT_ID
          }
        }
      }
    },
    fields: [...requiredFields, ...requiredTxFields, ...requiredSpanFields, ...optionalFields],
    sort: [{
      _score: 'asc'
    }, {
      _script: {
        type: 'number',
        script: {
          lang: 'painless',
          source: `$('${_apm.TRANSACTION_DURATION}', $('${_apm.SPAN_DURATION}', 0))`
        },
        order: 'desc'
      }
    }, {
      '@timestamp': 'asc'
    }, {
      _doc: 'asc'
    }]
  };
  const res = await apmEventClient.search('get_trace_docs', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    ...body
  });
  return {
    hits: res.hits.hits.map(hit => {
      var _hit$_source$span;
      const sort = hit.sort;
      const fields = (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(requiredFields);
      const spanLinks = 'span' in hit._source ? (_hit$_source$span = hit._source.span) === null || _hit$_source$span === void 0 ? void 0 : _hit$_source$span.links : (0, _utils2.mapOtelToSpanLink)({
        span_id: fields[_apm.OTEL_SPAN_LINKS_SPAN_ID],
        trace_id: fields[_apm.OTEL_SPAN_LINKS_TRACE_ID]
      });
      if (fields[_apm.PROCESSOR_EVENT] === _common.ProcessorEvent.span) {
        const {
          child,
          span,
          processor,
          ...spanEvent
        } = fields.requireFields(requiredSpanFields).unflatten();
        const spanWaterfallEvent = {
          ...spanEvent,
          processor: processor,
          span: {
            ...span,
            composite: span.composite ? span.composite : undefined,
            links: spanLinks
          },
          ...(child ? {
            child: child
          } : undefined)
        };
        return {
          sort,
          hit: spanWaterfallEvent
        };
      }
      const {
        span,
        processor,
        ...txEvent
      } = fields.requireFields(requiredTxFields).unflatten();
      const txWaterfallEvent = {
        ...txEvent,
        processor: processor,
        span: {
          ...span,
          links: spanLinks
        }
      };
      return {
        hit: txWaterfallEvent,
        sort
      };
    }),
    total: res.hits.total.value
  };
}