"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSecurityProductFeaturesConfigurator = void 0;
var _keys = require("@kbn/security-solution-features/keys");
var _config = require("@kbn/security-solution-features/config");
var _constants = require("@kbn/security-solution-features/constants");
var _common = require("@kbn/security-solution-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSecurityProductFeaturesConfigurator = enabledProductFeatureKeys => () => {
  return (0, _config.createEnabledProductFeaturesConfigMap)(securityProductFeaturesConfig, enabledProductFeatureKeys);
};

/**
 * Maps the ProductFeatures keys to Kibana privileges that will be merged
 * into the base privileges config for the Security app.
 *
 * Privileges can be added in different ways:
 * - `privileges`: the privileges that will be added directly into the main Security feature.
 * - `subFeatureIds`: the ids of the sub-features that will be added into the Security subFeatures entry.
 * - `subFeaturesPrivileges`: the privileges that will be added into the existing Security subFeature with the privilege `id` specified.
 */
exports.getSecurityProductFeaturesConfigurator = getSecurityProductFeaturesConfigurator;
const securityProductFeaturesConfig = {
  ..._config.securityDefaultProductFeaturesConfig,
  [_keys.ProductFeatureSecurityKey.endpointExceptions]: {
    privileges: {
      all: {
        ui: ['showEndpointExceptions', 'crudEndpointExceptions'],
        api: [`${_common.APP_ID}-showEndpointExceptions`, `${_common.APP_ID}-crudEndpointExceptions`]
      },
      read: {
        ui: ['showEndpointExceptions'],
        api: [`${_common.APP_ID}-showEndpointExceptions`]
      }
    }
  },
  [_keys.ProductFeatureSecurityKey.endpointArtifactManagement]: {
    subFeatureIds: [_keys.SecuritySubFeatureId.hostIsolationExceptionsBasic, _keys.SecuritySubFeatureId.trustedApplications, _keys.SecuritySubFeatureId.blocklist, _keys.SecuritySubFeatureId.eventFilters, _keys.SecuritySubFeatureId.globalArtifactManagement],
    baseFeatureConfigModifier: baseFeatureConfig => {
      var _baseFeatureConfig$pr, _baseFeatureConfig$pr2;
      if (!['siem', 'siemV2'].includes(baseFeatureConfig.id) || !((_baseFeatureConfig$pr = baseFeatureConfig.privileges) !== null && _baseFeatureConfig$pr !== void 0 && _baseFeatureConfig$pr.all.replacedBy) || !('default' in baseFeatureConfig.privileges.all.replacedBy)) {
        return baseFeatureConfig;
      }
      return {
        ...baseFeatureConfig,
        privileges: {
          ...baseFeatureConfig.privileges,
          all: {
            ...baseFeatureConfig.privileges.all,
            // overwriting siem:ALL role migration in siem and siemV2
            replacedBy: {
              default: baseFeatureConfig.privileges.all.replacedBy.default.map(privilegesPreference => {
                if (privilegesPreference.feature === _constants.SECURITY_FEATURE_ID_V3) {
                  return {
                    feature: _constants.SECURITY_FEATURE_ID_V3,
                    privileges: [
                    // Enabling sub-features toggle to show that Global Artifact Management is now provided to the user.
                    'minimal_all',
                    // Writing global (not per-policy) Artifacts is gated with Global Artifact Management:ALL starting with siemV3.
                    // Users who have been able to write ANY Artifact before are now granted with this privilege to keep existing behavior.
                    // This migration is for Endpoint Exceptions artifact in ESS offering, as it included in Security:ALL privilege.
                    'global_artifact_management_all']
                  };
                }
                return privilegesPreference;
              }),
              minimal: baseFeatureConfig.privileges.all.replacedBy.minimal.map(privilegesPreference => {
                if (privilegesPreference.feature === _constants.SECURITY_FEATURE_ID_V3) {
                  return {
                    feature: _constants.SECURITY_FEATURE_ID_V3,
                    privileges: ['minimal_all',
                    // on ESS, Endpoint Exception ALL is included in siem:MINIMAL_ALL
                    'global_artifact_management_all']
                  };
                }
                return privilegesPreference;
              })
            },
            api: [...((_baseFeatureConfig$pr2 = baseFeatureConfig.privileges.all.api) !== null && _baseFeatureConfig$pr2 !== void 0 ? _baseFeatureConfig$pr2 : []),
            // API access must be also added, as only UI privileges are copied when replacing a deprecated feature
            `${_common.APP_ID}-writeGlobalArtifacts`]
          }
        }
      };
    }
  }
};