"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsGetAPIRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _boom = require("@hapi/boom");
var _content_management = require("../../../../common/content_management");
var _constants = require("../../constants");
var _v = require("../../../content_management/v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsGetAPIRoute = (router, {
  contentManagement
}) => {
  const getRoute = router.get({
    path: `${_constants.PUBLIC_API_PATH}/visualizations/{id}`,
    access: _constants.PUBLIC_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Get Lens visualization',
    description: 'Get a Lens visualization from id.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  getRoute.addVersion({
    version: _constants.PUBLIC_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'The saved object id of a Lens visualization.'
            }
          })
        })
      },
      response: {
        200: {
          body: () => _v.lensSavedObjectSchema,
          description: 'Ok'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        404: {
          description: 'Resource not found'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    let result;
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _constants.PUBLIC_API_CONTENT_MANAGEMENT_VERSION);
    try {
      ({
        result
      } = await client.get(req.params.id));
    } catch (error) {
      if ((0, _boom.isBoom)(error)) {
        if (error.output.statusCode === 404) {
          return res.notFound({
            body: {
              message: `A Lens visualization with saved object id [${req.params.id}] was not found.`
            }
          });
        }
        if (error.output.statusCode === 403) {
          return res.forbidden();
        }
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
    return res.ok({
      body: result.item
    });
  });
};
exports.registerLensVisualizationsGetAPIRoute = registerLensVisualizationsGetAPIRoute;