"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withInferenceSpan = withInferenceSpan;
var _api = require("@opentelemetry/api");
var _rxjs = require("rxjs");
var _types = require("util/types");
var _lodash = require("lodash");
var _create_inference_active_span = require("./create_inference_active_span");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Wraps a callback in an active span. If the callback returns an Observable
 * or Promise, it will set the span status to the appropriate value when the
 * async operation completes.
 * @param options
 * @param cb
 */
function withInferenceSpan(options, cb) {
  const parentContext = _api.context.active();
  return (0, _create_inference_active_span.createActiveInferenceSpan)(options, span => {
    if (!span) {
      return cb();
    }
    try {
      const res = cb(span);
      if ((0, _rxjs.isObservable)(res)) {
        return withInferenceSpan$(span, parentContext, res);
      }
      if ((0, _types.isPromise)(res)) {
        return withInferenceSpanPromise(span, res);
      }
      span.setStatus({
        code: _api.SpanStatusCode.OK
      });
      span.end();
      return res;
    } catch (error) {
      span.recordException(error);
      span.setStatus({
        code: _api.SpanStatusCode.ERROR,
        message: error.message
      });
      span.end();
      throw error;
    }
  });
}
function withInferenceSpan$(span, parentContext, source$) {
  const ctx = _api.context.active();
  return new _rxjs.Observable(subscriber => {
    // Make sure anything that happens during this callback uses the context
    // that was active when this function was called
    const subscription = _api.context.with(ctx, () => {
      const end = (0, _lodash.once)(error => {
        if (span.isRecording()) {
          span.recordException(error);
          span.setStatus({
            code: _api.SpanStatusCode.ERROR,
            message: error.message
          });
          span.end();
        }
      });
      return source$.pipe((0, _rxjs.tap)({
        next: value => {
          subscriber.next(value);
        },
        error: error => {
          // Call span.end() and subscriber.error() in the parent context, to
          // ensure a span that gets created right after doesn't get created
          // as a child of this span, but as a child of its parent span.
          _api.context.with(parentContext, () => {
            end(error);
            subscriber.error(error);
          });
        }
      }), (0, _rxjs.switchMap)(value => {
        // unwraps observable -> observable | promise which is a use case for the
        // Observability AI Assistant in tool calling
        if ((0, _rxjs.isObservable)(value)) {
          return value;
        }
        if ((0, _types.isPromise)(value)) {
          return (0, _rxjs.from)(value);
        }
        return (0, _rxjs.of)(value);
      }), (0, _rxjs.ignoreElements)()).subscribe({
        error: error => {
          _api.context.with(parentContext, () => {
            end(error);
            subscriber.error(error);
          });
        },
        complete: () => {
          _api.context.with(parentContext, () => {
            span.setStatus({
              code: _api.SpanStatusCode.OK
            });
            span.end();
            subscriber.complete();
          });
        }
      });
    });
    return () => _api.context.with(parentContext, () => subscription.unsubscribe());
  });
}
function withInferenceSpanPromise(span, promise) {
  return promise.then(res => {
    span.setStatus({
      code: _api.SpanStatusCode.OK
    });
    span.end();
    return res;
  }).catch(error => {
    span.recordException(error);
    span.setStatus({
      code: _api.SpanStatusCode.ERROR,
      message: error.message
    });
    span.end();
    throw error;
  });
}