"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validationHasErrors = exports.validatePackagePolicyConfig = exports.validatePackagePolicy = exports.countValidationErrors = void 0;
var _std = require("@kbn/std");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _constants = require("../constants");
var _ = require(".");
var _policy_template = require("./policy_template");
var _is_valid_namespace = require("./is_valid_namespace");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validatePackageRequiredVars = (streamOrInput, requiredVars) => {
  const evaluatedRequiredVars = {};
  if (!requiredVars || !streamOrInput.vars || !streamOrInput.enabled) {
    return null;
  }
  let hasMetRequiredCriteria = false;
  for (const [requiredVarDefinitionName, requiredVarDefinitionConstraints] of Object.entries(requiredVars)) {
    var _evaluatedRequiredVar;
    evaluatedRequiredVars[requiredVarDefinitionName] = (requiredVarDefinitionConstraints === null || requiredVarDefinitionConstraints === void 0 ? void 0 : requiredVarDefinitionConstraints.map(constraint => {
      return {
        name: constraint.name,
        invalid: true
      };
    })) || [];
    if (evaluatedRequiredVars[requiredVarDefinitionName]) {
      requiredVarDefinitionConstraints.forEach(requiredCondition => {
        if (streamOrInput.vars && streamOrInput.vars[requiredCondition.name]) {
          const varItem = streamOrInput.vars[requiredCondition.name];
          if (varItem) {
            if (!requiredCondition.value && varItem.value || requiredCondition.value && varItem.value && requiredCondition.value === varItem.value) {
              evaluatedRequiredVars[requiredVarDefinitionName] = evaluatedRequiredVars[requiredVarDefinitionName].filter(item => item.name !== requiredCondition.name);
            }
          }
        }
      });
    }
    if (((_evaluatedRequiredVar = evaluatedRequiredVars[requiredVarDefinitionName]) === null || _evaluatedRequiredVar === void 0 ? void 0 : _evaluatedRequiredVar.length) === 0) {
      hasMetRequiredCriteria = true;
    }
  }
  return hasMetRequiredCriteria ? null : evaluatedRequiredVars;
};
const VALIDATE_DATASTREAMS_PERMISSION_REGEX = /^(logs)|(metrics)|(traces)|(synthetics)|(profiling)-(.*)$/;

/*
 * Returns validation information for a given package policy and package info
 * Note: this method assumes that `packagePolicy` is correctly structured for the given package
 */
const validatePackagePolicy = (packagePolicy, packageInfo, safeLoadYaml, spaceSettings) => {
  var _packageInfo$policy_t;
  const hasIntegrations = (0, _.doesPackageHaveIntegrations)(packageInfo);
  const validationResults = {
    name: null,
    description: null,
    namespace: null,
    additional_datastreams_permissions: null,
    inputs: {},
    vars: {}
  };
  if (!packagePolicy.name.trim()) {
    validationResults.name = [_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.nameRequiredErrorMessage', {
      defaultMessage: 'Name is required'
    })];
  }
  if (packagePolicy !== null && packagePolicy !== void 0 && packagePolicy.namespace) {
    const namespaceValidation = (0, _.isValidNamespace)(packagePolicy === null || packagePolicy === void 0 ? void 0 : packagePolicy.namespace, true, spaceSettings === null || spaceSettings === void 0 ? void 0 : spaceSettings.allowedNamespacePrefixes);
    if (!namespaceValidation.valid && namespaceValidation.error) {
      validationResults.namespace = [namespaceValidation.error];
    }
  }
  if (packagePolicy !== null && packagePolicy !== void 0 && packagePolicy.additional_datastreams_permissions) {
    validationResults.additional_datastreams_permissions = packagePolicy === null || packagePolicy === void 0 ? void 0 : packagePolicy.additional_datastreams_permissions.reduce((acc, additionalDatastreamsPermission) => {
      if (!additionalDatastreamsPermission.match(VALIDATE_DATASTREAMS_PERMISSION_REGEX)) {
        if (!acc) {
          acc = [];
        }
        acc.push(`${additionalDatastreamsPermission} is not valid, should match ${VALIDATE_DATASTREAMS_PERMISSION_REGEX.toString()}`);
      }
      return acc;
    }, null);
  }

  // Validate package-level vars
  const packageVarsByName = (0, _lodash.keyBy)(packageInfo.vars || [], 'name');
  const packageVars = Object.entries(packagePolicy.vars || {});
  if (packageVars.length) {
    validationResults.vars = packageVars.reduce((results, [name, varEntry]) => {
      results[name] = validatePackagePolicyConfig(varEntry, packageVarsByName[name], name, safeLoadYaml);
      return results;
    }, {});
  }
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t = packageInfo.policy_templates) !== null && _packageInfo$policy_t !== void 0 && _packageInfo$policy_t.length) || (0, _policy_template.packageHasNoPolicyTemplates)(packageInfo)) {
    validationResults.inputs = {};
    return validationResults;
  }

  // Build cache for fast var definition lookup
  const inputVarDefsByPolicyTemplateAndType = packageInfo.policy_templates.reduce((varDefs, policyTemplate) => {
    const inputs = (0, _.getNormalizedInputs)(policyTemplate);
    inputs.forEach(input => {
      const varDefKey = hasIntegrations ? `${policyTemplate.name}-${input.type}` : input.type;
      if ((input.vars || []).length) {
        varDefs[varDefKey] = (0, _lodash.keyBy)(input.vars || [], 'name');
      }
    });
    return varDefs;
  }, {});
  const inputRequiredVarsDefsByPolicyTemplateAndType = packageInfo.policy_templates.reduce((reqVarDefs, policyTemplate) => {
    const inputs = (0, _.getNormalizedInputs)(policyTemplate);
    inputs.forEach(input => {
      const requiredVarDefKey = hasIntegrations ? `${policyTemplate.name}-${input.type}` : input.type;
      if ((input.vars || []).length) {
        reqVarDefs[requiredVarDefKey] = input.required_vars;
      }
    });
    return reqVarDefs;
  }, {});
  const dataStreams = (0, _.getNormalizedDataStreams)(packageInfo);
  const streamsByDatasetAndInput = dataStreams.reduce((streams, dataStream) => {
    var _dataStream$streams;
    (_dataStream$streams = dataStream.streams) === null || _dataStream$streams === void 0 ? void 0 : _dataStream$streams.forEach(stream => {
      streams[`${dataStream.dataset}-${stream.input}`] = stream;
    });
    return streams;
  }, {});
  const streamVarDefsByDatasetAndInput = Object.entries(streamsByDatasetAndInput).reduce((varDefs, [path, stream]) => {
    varDefs[path] = (0, _lodash.keyBy)(stream.vars || [], 'name');
    return varDefs;
  }, {});
  const streamRequiredVarsDefsByDataAndInput = Object.entries(streamsByDatasetAndInput).reduce((reqVarDefs, [path, stream]) => {
    reqVarDefs[path] = stream.required_vars;
    return reqVarDefs;
  }, {});

  // Validate each package policy input with either its own var fields and stream vars
  packagePolicy.inputs.forEach(input => {
    if (!input.vars && !input.streams) {
      return;
    }
    const inputKey = hasIntegrations ? `${input.policy_template}-${input.type}` : input.type;
    const inputValidationResults = {
      vars: undefined,
      required_vars: undefined,
      streams: {}
    };

    // Validate input-level var fields
    const inputVars = Object.entries(input.vars || {});
    if (inputVars.length) {
      inputValidationResults.vars = inputVars.reduce((results, [name, configEntry]) => {
        var _inputVarDefsByPolicy;
        results[name] = input.enabled ? validatePackagePolicyConfig(configEntry, ((_inputVarDefsByPolicy = inputVarDefsByPolicyTemplateAndType[inputKey]) !== null && _inputVarDefsByPolicy !== void 0 ? _inputVarDefsByPolicy : {})[name], name, safeLoadYaml) : null;
        return results;
      }, {});
      const requiredVars = input.enabled && validatePackageRequiredVars(input, inputRequiredVarsDefsByPolicyTemplateAndType[inputKey]);
      if (requiredVars) {
        inputValidationResults.required_vars = requiredVars;
      } else {
        delete inputValidationResults.required_vars;
      }
    } else {
      delete inputValidationResults.vars;
      delete inputValidationResults.required_vars;
    }

    // Validate each input stream with var definitions
    if (input.streams.length) {
      input.streams.forEach(stream => {
        const streamValidationResults = {};
        const streamVarDefs = streamVarDefsByDatasetAndInput[`${stream.data_stream.dataset}-${input.type}`];
        if (streamVarDefs && Object.keys(streamVarDefs).length) {
          streamValidationResults.vars = Object.keys(streamVarDefs).reduce((results, name) => {
            var _stream$vars;
            const configEntry = stream === null || stream === void 0 ? void 0 : (_stream$vars = stream.vars) === null || _stream$vars === void 0 ? void 0 : _stream$vars[name];
            results[name] = input.enabled && stream.enabled ? validatePackagePolicyConfig(configEntry, streamVarDefs[name], name, safeLoadYaml, packageInfo.type) : null;
            return results;
          }, {});
        }
        if (stream.vars && stream.enabled) {
          const requiredVars = validatePackageRequiredVars(stream, streamRequiredVarsDefsByDataAndInput[`${stream.data_stream.dataset}-${input.type}`]);
          if (requiredVars) {
            streamValidationResults.required_vars = requiredVars;
          }
        }
        inputValidationResults.streams[stream.data_stream.dataset] = streamValidationResults;
      });
    } else {
      delete inputValidationResults.streams;
    }
    if (inputValidationResults.vars || inputValidationResults.streams) {
      validationResults.inputs[inputKey] = inputValidationResults;
    }
  });
  if (Object.entries(validationResults.inputs).length === 0) {
    validationResults.inputs = {};
  }
  return validationResults;
};
exports.validatePackagePolicy = validatePackagePolicy;
const validatePackagePolicyConfig = (configEntry, varDef, varName, safeLoadYaml, packageType) => {
  const errors = [];
  const value = configEntry === null || configEntry === void 0 ? void 0 : configEntry.value;
  let parsedValue = value;
  if (typeof value === 'string') {
    parsedValue = value.trim();
  }
  if (varDef === undefined) {
    // TODO return validation error here once https://github.com/elastic/kibana/issues/125655 is fixed
    // eslint-disable-next-line no-console
    console.debug(`No variable definition for ${varName} found`);
    return null;
  }
  if (varDef.required) {
    if (parsedValue === undefined || varDef.type === 'yaml' && parsedValue === '') {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.requiredErrorMessage', {
        defaultMessage: '{fieldName} is required',
        values: {
          fieldName: varDef.title || varDef.name
        }
      }));
    }
  }
  if (varDef.secret === true && parsedValue && parsedValue.isSecretRef === true) {
    if (!parsedValue.id && (!parsedValue.ids || parsedValue.ids.length === 0)) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidSecretReference', {
        defaultMessage: 'Secret reference is invalid, id or ids must be provided'
      }));
      return errors;
    }
    if (parsedValue.id && parsedValue.ids) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidSecretReference', {
        defaultMessage: 'Secret reference is invalid, id or ids cannot both be provided'
      }));
      return errors;
    }
    if (parsedValue.id && typeof parsedValue.id !== 'string') {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidSecretReference', {
        defaultMessage: 'Secret reference is invalid, id must be a string'
      }));
      return errors;
    }
    if (parsedValue.ids && !parsedValue.ids.every(id => typeof id === 'string')) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidSecretReference', {
        defaultMessage: 'Secret reference is invalid, ids must be an array of strings'
      }));
      return errors;
    }
    return null;
  }
  if (varDef.type === 'yaml') {
    try {
      parsedValue = safeLoadYaml(value);
    } catch (e) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidYamlFormatErrorMessage', {
        defaultMessage: 'Invalid YAML format'
      }));
    }
  }
  if (varDef.multi) {
    if (parsedValue && !Array.isArray(parsedValue)) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidArrayErrorMessage', {
        defaultMessage: 'Invalid format for {fieldName}: expected array',
        values: {
          fieldName: varDef.title || varDef.name
        }
      }));
      return errors;
    }
    if (varDef.required && Array.isArray(parsedValue)) {
      const hasEmptyString = varDef.type === 'text' && parsedValue.some(item => typeof item === 'string' && item.trim() === '');
      if (hasEmptyString || parsedValue.length === 0) {
        errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.requiredErrorMessage', {
          defaultMessage: '{fieldName} is required',
          values: {
            fieldName: varDef.title || varDef.name
          }
        }));
      }
    }
    if (varDef.type === 'text' && parsedValue) {
      const invalidStrings = parsedValue.filter(cand => /^[*&]/.test(cand));
      // only show one error if multiple strings in array are invalid
      if (invalidStrings.length > 0) {
        errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.quoteStringErrorMessage', {
          defaultMessage: 'Strings starting with special YAML characters like * or & need to be enclosed in double quotes.'
        }));
      }
    }
    if (varDef.type === 'integer' && parsedValue) {
      const invalidIntegers = parsedValue.filter(val => !Number.isInteger(Number(val)));
      // only show one error if multiple strings in array are invalid
      if (invalidIntegers.length > 0) {
        errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidIntegerMultiErrorMessage', {
          defaultMessage: 'Invalid integer'
        }));
      }
    }
    return errors.length ? errors : null;
  }
  if (varDef.type === 'text' && parsedValue && !Array.isArray(parsedValue)) {
    if (/^[*&]/.test(parsedValue)) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.quoteStringErrorMessage', {
        defaultMessage: 'Strings starting with special YAML characters like * or & need to be enclosed in double quotes.'
      }));
    }
  }
  if (varDef.type === 'bool' && parsedValue && !['true', 'false'].includes(parsedValue.toString())) {
    errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.boolValueError', {
      defaultMessage: 'Boolean values must be either true or false'
    }));
  }
  if (varDef.type === 'integer' && parsedValue && !Array.isArray(parsedValue)) {
    if (!Number.isInteger(Number(parsedValue))) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidIntegerErrorMessage', {
        defaultMessage: 'Invalid integer'
      }));
    }
  }
  if (varDef.type === 'select' && parsedValue !== undefined) {
    var _varDef$options;
    if (!((_varDef$options = varDef.options) !== null && _varDef$options !== void 0 && _varDef$options.map(o => o.value).includes(parsedValue))) {
      errors.push(_i18n.i18n.translate('xpack.fleet.packagePolicyValidation.invalidSelectValueErrorMessage', {
        defaultMessage: 'Invalid value for select type'
      }));
    }
  }
  if (varName === _constants.DATASET_VAR_NAME && packageType === 'input' && parsedValue !== undefined) {
    const {
      valid,
      error
    } = (0, _is_valid_namespace.isValidDataset)(parsedValue.dataset ? parsedValue.dataset : parsedValue, false);
    if (!valid && error) {
      errors.push(error);
    }
  }
  return errors.length ? errors : null;
};
exports.validatePackagePolicyConfig = validatePackagePolicyConfig;
const countValidationErrors = validationResults => {
  const requiredVarGroupErrorKeys = {};
  let otherErrors = 0;

  // Flatten validation results and map to retrieve required var group errors vs other errors
  // because required var groups should only count as 1 error
  const flattenedValidation = (0, _std.getFlattenedObject)(validationResults !== null && validationResults !== void 0 ? validationResults : {});
  Object.entries(flattenedValidation).forEach(([key, value]) => {
    if (key.startsWith('required_vars.')) {
      requiredVarGroupErrorKeys.required_vars = true;
    } else if (key.includes('.required_vars.')) {
      const groupKey = key.replace(/^(.*)\.required_vars\..*$/, '$1');
      requiredVarGroupErrorKeys[groupKey] = true;
    } else if (Boolean(value)) {
      otherErrors++;
    }
  });
  return otherErrors + Object.keys(requiredVarGroupErrorKeys).length;
};
exports.countValidationErrors = countValidationErrors;
const validationHasErrors = validationResults => {
  return countValidationErrors(validationResults) > 0;
};
exports.validationHasErrors = validationHasErrors;