"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getErrorCountByDocId = getErrorCountByDocId;
exports.getUnifiedTraceItems = getUnifiedTraceItems;
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _common = require("@kbn/observability-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _apm = require("../../../common/es_fields/apm");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _get_trace_items = require("./get_trace_items");
var _get_unified_trace_errors = require("./get_unified_trace_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fields = (0, _as_mutable_array.asMutableArray)(['@timestamp', 'trace.id', 'service.name']);
const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_NAME, _apm.DURATION, _apm.SPAN_DURATION, _apm.TRANSACTION_DURATION, _apm.TRANSACTION_ID, _apm.TRANSACTION_NAME, _apm.PROCESSOR_EVENT, _apm.PARENT_ID, _apm.STATUS_CODE, _apm.TIMESTAMP_US]);
function getErrorCountByDocId(unifiedTraceErrors) {
  const groupedErrorCountByDocId = {};
  function incrementErrorCount(id) {
    if (!groupedErrorCountByDocId[id]) {
      groupedErrorCountByDocId[id] = 0;
    }
    groupedErrorCountByDocId[id] += 1;
  }
  unifiedTraceErrors.apmErrors.forEach(doc => {
    var _doc$parent;
    return (_doc$parent = doc.parent) !== null && _doc$parent !== void 0 && _doc$parent.id ? incrementErrorCount(doc.parent.id) : undefined;
  });
  unifiedTraceErrors.unprocessedOtelErrors.forEach(doc => doc.id ? incrementErrorCount(doc.id) : undefined);
  return groupedErrorCountByDocId;
}

/**
 * Returns both APM documents and unprocessed OTEL spans
 */
async function getUnifiedTraceItems({
  apmEventClient,
  logsClient,
  maxTraceItemsFromUrlParam,
  traceId,
  start,
  end,
  config
}) {
  const maxTraceItems = maxTraceItemsFromUrlParam !== null && maxTraceItemsFromUrlParam !== void 0 ? maxTraceItemsFromUrlParam : config.ui.maxTraceItems;
  const size = Math.min(maxTraceItems, _get_trace_items.MAX_ITEMS_PER_PAGE);
  const unifiedTraceErrorsPromise = (0, _get_unified_trace_errors.getUnifiedTraceErrors)({
    apmEventClient,
    logsClient,
    traceId,
    start,
    end
  });
  const unifiedTracePromise = apmEventClient.search('get_unified_trace_items', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: true,
    size,
    query: {
      bool: {
        must: [{
          bool: {
            filter: [...(0, _server.termQuery)(_apm.TRACE_ID, traceId), ...(0, _server.rangeQuery)(start, end)],
            should: {
              exists: {
                field: _apm.PARENT_ID
              }
            }
          }
        }],
        should: [{
          terms: {
            [_apm.PROCESSOR_EVENT]: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
          }
        }, {
          bool: {
            must_not: {
              exists: {
                field: _apm.PROCESSOR_EVENT
              }
            }
          }
        }],
        minimum_should_match: 1
      }
    },
    fields: [...fields, ...optionalFields],
    sort: [{
      _score: 'asc'
    }, {
      _script: {
        type: 'number',
        script: {
          lang: 'painless',
          source: `$('${_apm.TRANSACTION_DURATION}', $('${_apm.SPAN_DURATION}', $('${_apm.DURATION}', 0)))`
        },
        order: 'desc'
      }
    }, {
      [_apm.AT_TIMESTAMP]: 'asc'
    }, {
      _doc: 'asc'
    }]
  }, {
    skipProcessorEventFilter: true
  });
  const [unifiedTraceErrors, unifiedTraceItems] = await Promise.all([unifiedTraceErrorsPromise, unifiedTracePromise]);
  const errorCountByDocId = getErrorCountByDocId(unifiedTraceErrors);
  return {
    traceItems: unifiedTraceItems.hits.hits.map(hit => {
      var _event$SPAN_DURATION, _event$SPAN_ID, _event$SPAN_NAME, _event$TIMESTAMP_US;
      const event = (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(fields);
      const apmDuration = (_event$SPAN_DURATION = event[_apm.SPAN_DURATION]) !== null && _event$SPAN_DURATION !== void 0 ? _event$SPAN_DURATION : event[_apm.TRANSACTION_DURATION];
      const id = (_event$SPAN_ID = event[_apm.SPAN_ID]) !== null && _event$SPAN_ID !== void 0 ? _event$SPAN_ID : event[_apm.TRANSACTION_ID];
      const name = (_event$SPAN_NAME = event[_apm.SPAN_NAME]) !== null && _event$SPAN_NAME !== void 0 ? _event$SPAN_NAME : event[_apm.TRANSACTION_NAME];
      if (!id || !name) {
        return undefined;
      }
      const docErrorCount = errorCountByDocId[id] || 0;
      const statusCode = event[_apm.STATUS_CODE];
      return {
        id,
        name,
        timestampUs: (_event$TIMESTAMP_US = event[_apm.TIMESTAMP_US]) !== null && _event$TIMESTAMP_US !== void 0 ? _event$TIMESTAMP_US : toMicroseconds(event[_apm.AT_TIMESTAMP]),
        traceId: event[_apm.TRACE_ID],
        duration: resolveDuration(apmDuration, event[_apm.DURATION]),
        hasError: docErrorCount > 0 || statusCode === 'Error',
        parentId: event[_apm.PARENT_ID],
        serviceName: event['service.name']
      };
    }).filter(_ => _),
    unifiedTraceErrors
  };
}

/**
 * Resolve either an APM or OTEL duration and if OTEL, format the duration from nanoseconds to microseconds.
 */
function resolveDuration(apmDuration, otelDuration) {
  if (apmDuration) {
    return apmDuration;
  }
  const duration = Array.isArray(otelDuration) ? otelDuration[0] : otelDuration ? parseFloat(otelDuration) : 0;
  return duration * 0.001;
}
const toMicroseconds = ts => new Date(ts).getTime() * 1000; // Convert ms to us