"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.researcherTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _run_researcher_agent = require("./run_researcher_agent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const researcherSchema = _zod.z.object({
  instructions: _zod.z.string().describe('Research instructions for the agent')
});
const researcherTool = () => {
  return {
    id: _onechatCommon.BuiltinToolIds.researcherAgent,
    description: `An agentic researcher tool to perform search and analysis tasks.

      Can be used to perform "deep search" tasks where a single query or search is not enough
      and where we need some kind of more in depth-research with multiple search requests and analysis.

      Example where the agent should be used:
        - "Summarize the changes between our previous and current work from home policy"
        - "Find the vulnerabilities involved in our latest alerts and gather information about them"
        - Any time the user explicitly asks to use this tool

      Example where the agent should not be used (in favor of more simple search tools):
        - "Show me the last 5 documents in the index 'foo'"
        - "Show me my latest alerts"

      Notes:
        - Please include all useful information in the instructions, as the agent has no other context. `,
    schema: researcherSchema,
    handler: async ({
      instructions
    }, context) => {
      const searchAgentResult = await (0, _run_researcher_agent.runResearcherAgent)({
        nextInput: {
          message: instructions
        },
        tools: context.toolProvider
      }, context);
      return {
        answer: searchAgentResult.round.assistantResponse.message
      };
    },
    meta: {
      tags: [_onechatCommon.BuiltinTags.retrieval]
    }
  };
};
exports.researcherTool = researcherTool;