"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _actions = require("./actions");
var _prompts = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const StateAnnotation = _langgraph.Annotation.Root({
  // inputs
  initialMessages: (0, _langgraph.Annotation)({
    reducer: (current, next) => {
      return [...current, ...next];
    },
    default: () => []
  }),
  // outputs
  addedMessages: (0, _langgraph.Annotation)({
    reducer: (current, next) => {
      return [...current, ...next];
    },
    default: () => []
  })
});
const createAgentGraph = async ({
  chatModel,
  tools
}) => {
  const toolNode = new _prebuilt.ToolNode(tools);
  const model = chatModel.bindTools(tools).withConfig({
    tags: ['onechat-agent']
  });
  const reason = async state => {
    const response = await model.invoke((0, _prompts.getReasoningPrompt)({
      messages: [...state.initialMessages, ...state.addedMessages]
    }));
    const reasoningEvent = {
      type: 'reasoning',
      reasoning: (0, _langchain.extractTextContent)(response)
    };
    return {
      addedMessages: [reasoningEvent]
    };
  };
  const act = async state => {
    const actModel = chatModel.bindTools(tools).withConfig({
      tags: ['reasoning:act']
    });
    const response = await actModel.invoke((0, _prompts.getActPrompt)({
      initialMessages: state.initialMessages,
      addedMessages: state.addedMessages
    }));
    return {
      addedMessages: [response]
    };
  };
  const shouldContinue = async state => {
    var _lastMessage$tool_cal;
    const messages = state.addedMessages.filter(_actions.isMessage);
    const lastMessage = messages[messages.length - 1];
    if (lastMessage && (_lastMessage$tool_cal = lastMessage.tool_calls) !== null && _lastMessage$tool_cal !== void 0 && _lastMessage$tool_cal.length) {
      return 'tools';
    }
    return '__end__';
  };
  const toolHandler = async state => {
    const toolNodeResult = await toolNode.invoke(state.addedMessages);
    return {
      addedMessages: [...toolNodeResult]
    };
  };

  // note: the node names are used in the event convertion logic, they should *not* be changed
  const graph = new _langgraph.StateGraph(StateAnnotation).addNode('reason', reason).addNode('act', act).addNode('tools', toolHandler).addEdge('__start__', 'reason').addEdge('reason', 'act').addEdge('tools', 'reason').addConditionalEdges('act', shouldContinue, {
    tools: 'tools',
    __end__: '__end__'
  }).compile();
  return graph;
};
exports.createAgentGraph = createAgentGraph;