"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.tasks = void 0;
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/observability-plugin/common");
var _crypto = require("crypto");
var _lodash = require("lodash");
var _server = require("@kbn/apm-sources-access-plugin/server");
var _agent_names = require("@kbn/elastic-agent-utils/src/agent_names");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _apm = require("../../../../common/es_fields/apm");
var _service_groups = require("../../../../common/service_groups");
var _as_mutable_array = require("../../../../common/utils/as_mutable_array");
var _rollup = require("../../../../common/rollup");
var _custom_dashboards = require("../../../../common/custom_dashboards");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TIME_RANGES = ['1d', 'all'];
const AGENT_NAMES_WITHOUT_OTEL = (0, _lodash.without)(_agent_names.AGENT_NAMES, ..._agent_names.OPEN_TELEMETRY_AGENT_NAMES);
const range1d = {
  range: {
    '@timestamp': {
      gte: 'now-1d'
    }
  }
};
const timeout = '5m';
const tasks = exports.tasks = [{
  name: 'aggregated_transactions',
  // Record the number of metric documents we can expect in different scenarios. We simulate this by requesting data for 1m,
  // adding a composite aggregation on a number of fields and counting the number of buckets. The resulting count is an
  // approximation of the amount of metric documents that will be created. We record both the expected metric document count plus
  // the transaction count for that time range.
  executor: async ({
    indices,
    telemetryClient
  }) => {
    async function getBucketCountFromPaginatedQuery(sources, prevResult, after) {
      // eslint-disable-next-line @typescript-eslint/naming-convention
      let {
        expected_metric_document_count
      } = prevResult !== null && prevResult !== void 0 ? prevResult : {
        transaction_count: 0,
        expected_metric_document_count: 0
      };
      const params = {
        index: [indices.transaction],
        track_total_hits: true,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [{
              term: {
                [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
              }
            }, {
              range: {
                '@timestamp': {
                  gte: start,
                  lt: end
                }
              }
            }]
          }
        },
        aggs: {
          transaction_metric_groups: {
            composite: {
              ...(after ? {
                after
              } : {}),
              size: 10000,
              sources: sources.map((source, index) => {
                return {
                  [index]: source
                };
              })
            }
          }
        }
      };
      const result = await telemetryClient.search(params);
      let nextAfter;
      if (result.aggregations) {
        nextAfter = result.aggregations.transaction_metric_groups.after_key;
        expected_metric_document_count += result.aggregations.transaction_metric_groups.buckets.length;
      }
      const transactionCount = result.hits.total.value;
      if (nextAfter) {
        return await getBucketCountFromPaginatedQuery(sources, {
          expected_metric_document_count,
          transaction_count: transactionCount
        }, nextAfter);
      }
      return {
        expected_metric_document_count,
        transaction_count: transactionCount,
        ratio: expected_metric_document_count / transactionCount
      };
    }

    // fixed date range for reliable results
    const lastTransaction = (await telemetryClient.search({
      index: indices.transaction,
      timeout,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
            }
          }]
        }
      },
      size: 1,
      track_total_hits: false,
      sort: {
        [_apm.AT_TIMESTAMP]: 'desc'
      },
      fields: [_apm.AT_TIMESTAMP]
    })).hits.hits[0];
    if (!lastTransaction) {
      return {};
    }
    const end = new Date(lastTransaction.fields[_apm.AT_TIMESTAMP][0]).getTime() - 5 * 60 * 1000;
    const start = end - 60 * 1000;
    const simpleTermFields = [_apm.TRANSACTION_NAME, _apm.TRANSACTION_RESULT, _apm.TRANSACTION_TYPE, _apm.AGENT_NAME, _apm.SERVICE_ENVIRONMENT, _apm.SERVICE_VERSION, _apm.HOST_NAME, _apm.CONTAINER_ID, _apm.KUBERNETES_POD_NAME].map(field => ({
      terms: {
        field,
        missing_bucket: true
      }
    }));
    const observerHostname = {
      terms: {
        field: _apm.OBSERVER_HOSTNAME,
        missing_bucket: true
      }
    };
    const baseFields = [...simpleTermFields,
    // user_agent.name only for page-load transactions
    {
      terms: {
        script: `
              if ($('transaction.type', '') == 'page-load') {
                return $('user_agent.name', null);
              }
              return null;
            `,
        missing_bucket: true
      }
    },
    // transaction.root
    {
      terms: {
        script: `return $('parent.id', '') == ''`,
        missing_bucket: true
      }
    }];
    const results = {
      current_implementation: await getBucketCountFromPaginatedQuery([...baseFields, observerHostname]),
      with_country: await getBucketCountFromPaginatedQuery([...baseFields, observerHostname, {
        terms: {
          script: `
                if ($('transaction.type', '') == 'page-load') {
                  return $('client.geo.country_iso_code', null);
                }
                return null;
              `,
          missing_bucket: true
        }
      }]),
      no_observer_name: await getBucketCountFromPaginatedQuery(baseFields)
    };
    return {
      aggregated_transactions: results
    };
  }
}, {
  name: 'cloud',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    function getBucketKeys({
      buckets
    }) {
      return buckets.map(bucket => bucket.key);
    }
    const az = 'availability_zone';
    const region = 'region';
    const provider = 'provider';
    const response = await telemetryClient.search({
      index: [indices.error, indices.metric, indices.span, indices.transaction],
      track_total_hits: false,
      size: 0,
      timeout,
      aggs: {
        [az]: {
          terms: {
            field: _apm.CLOUD_AVAILABILITY_ZONE
          }
        },
        [provider]: {
          terms: {
            field: _apm.CLOUD_PROVIDER
          }
        },
        [region]: {
          terms: {
            field: _apm.CLOUD_REGION
          }
        }
      }
    });
    const {
      aggregations
    } = response;
    if (!aggregations) {
      return {
        cloud: {
          [az]: [],
          [provider]: [],
          [region]: []
        }
      };
    }
    const cloud = {
      [az]: getBucketKeys(aggregations[az]),
      [provider]: getBucketKeys(aggregations[provider]),
      [region]: getBucketKeys(aggregations[region])
    };
    return {
      cloud
    };
  }
}, {
  name: 'host',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    function getBucketKeys({
      buckets
    }) {
      return buckets.map(bucket => bucket.key);
    }
    const response = await telemetryClient.search({
      index: [indices.error, indices.metric, indices.span, indices.transaction],
      track_total_hits: false,
      size: 0,
      timeout,
      aggs: {
        platform: {
          terms: {
            field: _apm.HOST_OS_PLATFORM
          }
        }
      }
    });
    const {
      aggregations
    } = response;
    if (!aggregations) {
      return {
        host: {
          os: {
            platform: []
          }
        }
      };
    }
    const host = {
      os: {
        platform: getBucketKeys(aggregations.platform)
      }
    };
    return {
      host
    };
  }
}, {
  name: 'environments',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4;
    const response = await telemetryClient.search({
      index: [indices.transaction],
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [range1d]
        }
      },
      aggs: {
        environments: {
          terms: {
            field: _apm.SERVICE_ENVIRONMENT,
            size: 5
          }
        },
        service_environments: {
          composite: {
            size: 1000,
            sources: (0, _as_mutable_array.asMutableArray)([{
              [_apm.SERVICE_ENVIRONMENT]: {
                terms: {
                  field: _apm.SERVICE_ENVIRONMENT,
                  missing_bucket: true
                }
              }
            }, {
              [_apm.SERVICE_NAME]: {
                terms: {
                  field: _apm.SERVICE_NAME
                }
              }
            }])
          }
        }
      }
    });
    const topEnvironments = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.environments.buckets.map(bucket => bucket.key)) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
    const serviceEnvironments = {};
    const buckets = (_response$aggregation3 = (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.service_environments.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
    buckets.forEach(bucket => {
      var _serviceEnvironments$;
      const serviceName = bucket.key['service.name'];
      const environment = bucket.key['service.environment'];
      const environments = (_serviceEnvironments$ = serviceEnvironments[serviceName]) !== null && _serviceEnvironments$ !== void 0 ? _serviceEnvironments$ : [];
      serviceEnvironments[serviceName] = environments.concat(environment);
    });
    const servicesWithoutEnvironment = Object.keys((0, _lodash.pickBy)(serviceEnvironments, environments => environments.includes(null)));
    const servicesWithMultipleEnvironments = Object.keys((0, _lodash.pickBy)(serviceEnvironments, environments => environments.length > 1));
    return {
      environments: {
        services_without_environment: servicesWithoutEnvironment.length,
        services_with_multiple_environments: servicesWithMultipleEnvironments.length,
        top_environments: topEnvironments
      }
    };
  }
}, {
  name: 'processor_events',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const indicesByProcessorEvent = {
      error: indices.error,
      metric: indices.metric,
      span: indices.span,
      transaction: indices.transaction,
      onboarding: indices.onboarding
    };
    const events = Object.keys(indicesByProcessorEvent);
    const jobs = events.flatMap(processorEvent => TIME_RANGES.map(timeRange => ({
      processorEvent,
      timeRange
    })));
    const allData = await jobs.reduce((prevJob, current) => {
      return prevJob.then(async data => {
        var _retainmentResponse$h;
        const {
          processorEvent,
          timeRange
        } = current;
        const totalHitsResponse = await telemetryClient.search({
          index: indicesByProcessorEvent[processorEvent],
          size: 0,
          track_total_hits: true,
          timeout,
          query: {
            bool: {
              filter: [{
                term: {
                  [_apm.PROCESSOR_EVENT]: processorEvent
                }
              }, ...(timeRange === '1d' ? [range1d] : [])]
            }
          }
        });
        const retainmentResponse = timeRange === 'all' ? await telemetryClient.search({
          index: indicesByProcessorEvent[processorEvent],
          size: 10,
          track_total_hits: false,
          timeout,
          query: {
            bool: {
              filter: [{
                term: {
                  [_apm.PROCESSOR_EVENT]: processorEvent
                }
              }]
            }
          },
          sort: {
            [_apm.AT_TIMESTAMP]: 'asc'
          },
          fields: [_apm.AT_TIMESTAMP]
        }) : null;
        const event = retainmentResponse === null || retainmentResponse === void 0 ? void 0 : (_retainmentResponse$h = retainmentResponse.hits.hits[0]) === null || _retainmentResponse$h === void 0 ? void 0 : _retainmentResponse$h.fields;
        return (0, _lodash.merge)({}, data, {
          counts: {
            [processorEvent]: {
              [timeRange]: totalHitsResponse.hits.total.value
            }
          },
          ...(event ? {
            retainment: {
              [processorEvent]: {
                ms: new Date().getTime() - new Date(event[_apm.AT_TIMESTAMP][0]).getTime()
              }
            }
          } : {})
        });
      });
    }, Promise.resolve({}));
    return allData;
  }
}, {
  name: 'agent_configuration',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const agentConfigurationCount = await telemetryClient.search({
      index: _server.APM_AGENT_CONFIGURATION_INDEX,
      size: 0,
      timeout,
      track_total_hits: true
    });
    return {
      counts: {
        agent_configuration: {
          all: agentConfigurationCount.hits.total.value
        }
      }
    };
  }
}, {
  name: 'global_labels',
  executor: async ({
    telemetryClient
  }) => {
    var _response$fields;
    const metricConsistingGlobalLabels = ['service_summary', 'service_transaction', 'transaction', 'service_destination'];
    const index = metricConsistingGlobalLabels.map(metric => `metrics-apm.${metric}*`).join(',');
    const response = await telemetryClient.fieldCaps({
      index,
      fields: 'labels.*',
      expand_wildcards: 'all',
      index_filter: range1d
    });
    const globalLabelCount = Object.keys(response.fields).length;

    // Skip the top level Labels field which is sometimes present in the response
    const count = (_response$fields = response.fields) !== null && _response$fields !== void 0 && _response$fields.labels ? globalLabelCount - 1 : globalLabelCount;
    return {
      counts: {
        global_labels: {
          '1d': count
        }
      }
    };
  }
}, {
  name: 'services',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _services$hits, _services$hits$total;
    const servicesPerAgentExcludingOtel = await AGENT_NAMES_WITHOUT_OTEL.reduce((prevJob, agentName) => {
      return prevJob.then(async data => {
        var _response$aggregation5;
        const response = await telemetryClient.search({
          index: [indices.error, indices.span, indices.metric, indices.transaction],
          size: 0,
          track_total_hits: false,
          timeout,
          query: {
            bool: {
              filter: [{
                term: {
                  [_apm.AGENT_NAME]: agentName
                }
              }, range1d]
            }
          },
          aggs: {
            services: {
              cardinality: {
                field: _apm.SERVICE_NAME
              }
            }
          }
        });
        data[agentName] = ((_response$aggregation5 = response.aggregations) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.services.value) || 0;
        return data;
      });
    }, Promise.resolve({}));
    const servicesPerOtelAgents = await _agent_names.OPEN_TELEMETRY_BASE_AGENT_NAMES.reduce((prevJob, baseAgentName) => {
      return prevJob.then(async accData => {
        const response = await telemetryClient.search({
          index: [indices.error, indices.span, indices.metric, indices.transaction],
          size: 0,
          track_total_hits: false,
          timeout,
          query: {
            bool: {
              filter: [{
                prefix: {
                  [_apm.AGENT_NAME]: baseAgentName
                }
              }, range1d]
            }
          },
          aggs: {
            agent_name: {
              terms: {
                field: _apm.AGENT_NAME,
                size: 1000
              },
              aggs: {
                services: {
                  cardinality: {
                    field: _apm.SERVICE_NAME
                  }
                }
              }
            }
          }
        });
        const aggregatedServices = {};
        for (const bucket of (_response$aggregation6 = (_response$aggregation7 = response.aggregations) === null || _response$aggregation7 === void 0 ? void 0 : _response$aggregation7.agent_name.buckets) !== null && _response$aggregation6 !== void 0 ? _response$aggregation6 : []) {
          var _response$aggregation6, _response$aggregation7;
          const fullAgentName = bucket.key;
          aggregatedServices[fullAgentName] = bucket.services.value || 0;
        }
        return {
          ...accData,
          ...aggregatedServices
        };
      });
    }, Promise.resolve({}));
    const services = await telemetryClient.search({
      index: [indices.error, indices.span, indices.metric, indices.transaction],
      size: 0,
      track_total_hits: true,
      terminate_after: 1,
      query: {
        bool: {
          filter: [{
            exists: {
              field: _apm.SERVICE_NAME
            }
          }, range1d]
        }
      },
      timeout
    });
    const servicesPerAgents = {
      ...servicesPerAgentExcludingOtel,
      ...servicesPerOtelAgents
    };
    return {
      has_any_services_per_official_agent: (0, _lodash.sum)(Object.values(servicesPerAgents)) > 0,
      has_any_services: (services === null || services === void 0 ? void 0 : (_services$hits = services.hits) === null || _services$hits === void 0 ? void 0 : (_services$hits$total = _services$hits.total) === null || _services$hits$total === void 0 ? void 0 : _services$hits$total.value) > 0,
      services_per_agent: servicesPerAgents
    };
  }
}, {
  name: 'versions',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$hits$hits$, _event$observer;
    const response = await telemetryClient.search({
      index: [indices.transaction, indices.span, indices.error],
      terminate_after: 1,
      query: {
        exists: {
          field: 'observer.version'
        }
      },
      track_total_hits: false,
      size: 1,
      timeout,
      sort: {
        '@timestamp': 'desc'
      },
      fields: (0, _as_mutable_array.asMutableArray)([_apm.OBSERVER_VERSION])
    });
    const event = (0, _utils.unflattenKnownApmEventFields)((_response$hits$hits$ = response.hits.hits[0]) === null || _response$hits$hits$ === void 0 ? void 0 : _response$hits$hits$.fields);
    if (!event || !((_event$observer = event.observer) !== null && _event$observer !== void 0 && _event$observer.version)) {
      return {};
    }
    const [major, minor, patch] = event.observer.version.split('.').map(part => Number(part));
    return {
      version: {
        apm_server: {
          major,
          minor,
          patch
        }
      }
    };
  }
}, {
  name: 'groupings',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _await$telemetryClien, _await$telemetryClien2, _await$telemetryClien3, _await$telemetryClien4, _await$telemetryClien5, _servicesAndEnvironme, _servicesAndEnvironme2;
    const errorGroupsCount = (_await$telemetryClien = (await telemetryClient.search({
      index: indices.error,
      size: 0,
      timeout,
      track_total_hits: false,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.error
            }
          }, range1d]
        }
      },
      aggs: {
        top_service: {
          terms: {
            field: _apm.SERVICE_NAME,
            order: {
              error_groups: 'desc'
            },
            size: 1
          },
          aggs: {
            error_groups: {
              cardinality: {
                field: _apm.ERROR_GROUP_ID
              }
            }
          }
        }
      }
    })).aggregations) === null || _await$telemetryClien === void 0 ? void 0 : (_await$telemetryClien2 = _await$telemetryClien.top_service.buckets[0]) === null || _await$telemetryClien2 === void 0 ? void 0 : _await$telemetryClien2.error_groups.value;
    const transactionGroupsCount = (_await$telemetryClien3 = (await telemetryClient.search({
      index: indices.transaction,
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
            }
          }, range1d]
        }
      },
      aggs: {
        top_service: {
          terms: {
            field: _apm.SERVICE_NAME,
            order: {
              transaction_groups: 'desc'
            },
            size: 1
          },
          aggs: {
            transaction_groups: {
              cardinality: {
                field: _apm.TRANSACTION_NAME
              }
            }
          }
        }
      }
    })).aggregations) === null || _await$telemetryClien3 === void 0 ? void 0 : (_await$telemetryClien4 = _await$telemetryClien3.top_service.buckets[0]) === null || _await$telemetryClien4 === void 0 ? void 0 : _await$telemetryClien4.transaction_groups.value;
    const tracesPerDayCount = (await telemetryClient.search({
      index: indices.transaction,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
            }
          }, range1d],
          must_not: {
            exists: {
              field: _apm.PARENT_ID
            }
          }
        }
      },
      track_total_hits: true,
      size: 0,
      timeout
    })).hits.total.value;
    const servicesAndEnvironmentsCount = await telemetryClient.search({
      index: [indices.transaction, indices.error, indices.metric],
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [range1d]
        }
      },
      aggs: {
        service_name: {
          cardinality: {
            field: _apm.SERVICE_NAME
          }
        },
        service_environments: {
          cardinality: {
            field: _apm.SERVICE_ENVIRONMENT
          }
        }
      }
    });
    const spanDestinationServiceResourceCount = (_await$telemetryClien5 = (await telemetryClient.search({
      index: indices.transaction,
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.span
            }
          }, range1d]
        }
      },
      aggs: {
        span_destination_service_resource: {
          cardinality: {
            field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE
          }
        }
      }
    })).aggregations) === null || _await$telemetryClien5 === void 0 ? void 0 : _await$telemetryClien5.span_destination_service_resource.value;
    return {
      counts: {
        max_error_groups_per_service: {
          '1d': errorGroupsCount || 0
        },
        max_transaction_groups_per_service: {
          '1d': transactionGroupsCount || 0
        },
        traces: {
          '1d': tracesPerDayCount || 0
        },
        services: {
          '1d': ((_servicesAndEnvironme = servicesAndEnvironmentsCount.aggregations) === null || _servicesAndEnvironme === void 0 ? void 0 : _servicesAndEnvironme.service_name.value) || 0
        },
        environments: {
          '1d': ((_servicesAndEnvironme2 = servicesAndEnvironmentsCount.aggregations) === null || _servicesAndEnvironme2 === void 0 ? void 0 : _servicesAndEnvironme2.service_environments.value) || 0
        },
        span_destination_service_resource: {
          '1d': spanDestinationServiceResourceCount || 0
        }
      }
    };
  }
}, {
  name: 'integrations',
  executor: async ({
    telemetryClient
  }) => {
    var _response$body$count, _response$body;
    const apmJobs = ['apm-*', '*-high_mean_response_time'];
    const response = await telemetryClient.transportRequest({
      method: 'get',
      path: `/_ml/anomaly_detectors/${apmJobs.join(',')}`
    });
    return {
      integrations: {
        ml: {
          all_jobs_count: (_response$body$count = (_response$body = response.body) === null || _response$body === void 0 ? void 0 : _response$body.count) !== null && _response$body$count !== void 0 ? _response$body$count : 0
        }
      }
    };
  }
}, {
  name: 'agents',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const size = 3;
    const toComposite = (outerKey, innerKey) => `${outerKey}/${innerKey}`;
    const agentNameAggs = {
      [_apm.AGENT_ACTIVATION_METHOD]: {
        terms: {
          field: _apm.AGENT_ACTIVATION_METHOD,
          size
        }
      },
      [_apm.AGENT_VERSION]: {
        terms: {
          field: _apm.AGENT_VERSION,
          size
        }
      },
      [_apm.SERVICE_FRAMEWORK_NAME]: {
        terms: {
          field: _apm.SERVICE_FRAMEWORK_NAME,
          size
        },
        aggs: {
          [_apm.SERVICE_FRAMEWORK_VERSION]: {
            terms: {
              field: _apm.SERVICE_FRAMEWORK_VERSION,
              size
            }
          }
        }
      },
      [_apm.SERVICE_FRAMEWORK_VERSION]: {
        terms: {
          field: _apm.SERVICE_FRAMEWORK_VERSION,
          size
        }
      },
      [_apm.SERVICE_LANGUAGE_NAME]: {
        terms: {
          field: _apm.SERVICE_LANGUAGE_NAME,
          size
        },
        aggs: {
          [_apm.SERVICE_LANGUAGE_VERSION]: {
            terms: {
              field: _apm.SERVICE_LANGUAGE_VERSION,
              size
            }
          }
        }
      },
      [_apm.SERVICE_LANGUAGE_VERSION]: {
        terms: {
          field: _apm.SERVICE_LANGUAGE_VERSION,
          size
        }
      },
      [_apm.SERVICE_RUNTIME_NAME]: {
        terms: {
          field: _apm.SERVICE_RUNTIME_NAME,
          size
        },
        aggs: {
          [_apm.SERVICE_RUNTIME_VERSION]: {
            terms: {
              field: _apm.SERVICE_RUNTIME_VERSION,
              size
            }
          }
        }
      },
      [_apm.SERVICE_RUNTIME_VERSION]: {
        terms: {
          field: _apm.SERVICE_RUNTIME_VERSION,
          size
        }
      }
    };
    const agentDataWithoutOtel = await AGENT_NAMES_WITHOUT_OTEL.reduce(async (prevJob, agentName) => {
      const data = await prevJob;
      const response = await telemetryClient.search({
        index: [indices.error, indices.metric, indices.transaction],
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [{
              term: {
                [_apm.AGENT_NAME]: agentName
              }
            }, range1d]
          }
        },
        sort: {
          '@timestamp': 'desc'
        },
        aggs: agentNameAggs
      });
      const {
        aggregations
      } = response;
      if (!aggregations) {
        return data;
      }
      data[agentName] = {
        agent: {
          activation_method: aggregations[_apm.AGENT_ACTIVATION_METHOD].buckets.map(bucket => bucket.key).slice(0, size),
          version: aggregations[_apm.AGENT_VERSION].buckets.map(bucket => bucket.key)
        },
        service: {
          framework: {
            name: aggregations[_apm.SERVICE_FRAMEWORK_NAME].buckets.map(bucket => bucket.key).slice(0, size),
            version: aggregations[_apm.SERVICE_FRAMEWORK_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
            composite: (0, _lodash.sortBy)((0, _lodash.flatten)(aggregations[_apm.SERVICE_FRAMEWORK_NAME].buckets.map(bucket => bucket[_apm.SERVICE_FRAMEWORK_VERSION].buckets.map(versionBucket => ({
              doc_count: versionBucket.doc_count,
              name: toComposite(bucket.key, versionBucket.key)
            })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
          },
          language: {
            name: aggregations[_apm.SERVICE_LANGUAGE_NAME].buckets.map(bucket => bucket.key).slice(0, size),
            version: aggregations[_apm.SERVICE_LANGUAGE_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
            composite: (0, _lodash.sortBy)((0, _lodash.flatten)(aggregations[_apm.SERVICE_LANGUAGE_NAME].buckets.map(bucket => bucket[_apm.SERVICE_LANGUAGE_VERSION].buckets.map(versionBucket => ({
              doc_count: versionBucket.doc_count,
              name: toComposite(bucket.key, versionBucket.key)
            })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
          },
          runtime: {
            name: aggregations[_apm.SERVICE_RUNTIME_NAME].buckets.map(bucket => bucket.key).slice(0, size),
            version: aggregations[_apm.SERVICE_RUNTIME_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
            composite: (0, _lodash.sortBy)((0, _lodash.flatten)(aggregations[_apm.SERVICE_RUNTIME_NAME].buckets.map(bucket => bucket[_apm.SERVICE_RUNTIME_VERSION].buckets.map(versionBucket => ({
              doc_count: versionBucket.doc_count,
              name: toComposite(bucket.key, versionBucket.key)
            })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
          }
        }
      };
      return data;
    }, Promise.resolve({}));
    const agentDataWithOtel = await _agent_names.OPEN_TELEMETRY_BASE_AGENT_NAMES.reduce(async (prevJob, baseAgentName) => {
      const data = await prevJob;
      const response = await telemetryClient.search({
        index: [indices.error, indices.metric, indices.transaction],
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [{
              prefix: {
                [_apm.AGENT_NAME]: baseAgentName
              }
            }, range1d]
          }
        },
        sort: {
          '@timestamp': 'desc'
        },
        aggs: {
          agent_name: {
            terms: {
              field: _apm.AGENT_NAME,
              size: 1000
            },
            aggs: agentNameAggs
          }
        }
      });
      if (!response.aggregations) {
        return data;
      }
      const dynamicAgentData = {};
      for (const agentBucket of response.aggregations.agent_name.buckets) {
        const agentKey = agentBucket.key;
        dynamicAgentData[agentKey] = {
          agent: {
            activation_method: agentBucket[_apm.AGENT_ACTIVATION_METHOD].buckets.map(bucket => bucket.key).slice(0, size),
            version: agentBucket[_apm.AGENT_VERSION].buckets.map(bucket => bucket.key)
          },
          service: {
            framework: {
              name: agentBucket[_apm.SERVICE_FRAMEWORK_NAME].buckets.map(bucket => bucket.key).slice(0, size),
              version: agentBucket[_apm.SERVICE_FRAMEWORK_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
              composite: (0, _lodash.sortBy)((0, _lodash.flatten)(agentBucket[_apm.SERVICE_FRAMEWORK_NAME].buckets.map(fwBucket => fwBucket[_apm.SERVICE_FRAMEWORK_VERSION].buckets.map(versionBucket => ({
                doc_count: versionBucket.doc_count,
                name: `${fwBucket.key}/${versionBucket.key}`
              })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
            },
            language: {
              name: agentBucket[_apm.SERVICE_LANGUAGE_NAME].buckets.map(bucket => bucket.key).slice(0, size),
              version: agentBucket[_apm.SERVICE_LANGUAGE_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
              composite: (0, _lodash.sortBy)((0, _lodash.flatten)(agentBucket[_apm.SERVICE_LANGUAGE_NAME].buckets.map(langBucket => langBucket[_apm.SERVICE_LANGUAGE_VERSION].buckets.map(versionBucket => ({
                doc_count: versionBucket.doc_count,
                name: `${langBucket.key}/${versionBucket.key}`
              })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
            },
            runtime: {
              name: agentBucket[_apm.SERVICE_RUNTIME_NAME].buckets.map(bucket => bucket.key).slice(0, size),
              version: agentBucket[_apm.SERVICE_RUNTIME_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
              composite: (0, _lodash.sortBy)((0, _lodash.flatten)(agentBucket[_apm.SERVICE_RUNTIME_NAME].buckets.map(runtimeBucket => runtimeBucket[_apm.SERVICE_RUNTIME_VERSION].buckets.map(versionBucket => ({
                doc_count: versionBucket.doc_count,
                name: `${runtimeBucket.key}/${versionBucket.key}`
              })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
            }
          }
        };
      }
      return {
        ...data,
        ...dynamicAgentData
      };
    }, Promise.resolve({}));
    return {
      agents: {
        ...agentDataWithoutOtel,
        ...agentDataWithOtel
      }
    };
  }
}, {
  name: 'indices_stats',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _metricIndicesRespons, _metricIndicesRespons2, _metricIndicesRespons3, _metricIndicesRespons4, _metricIndicesRespons5, _metricIndicesRespons6, _metricIndicesRespons7, _metricIndicesRespons8, _metricIndicesRespons9, _metricIndicesRespons10, _tracesIndicesRespons, _tracesIndicesRespons2, _tracesIndicesRespons3, _tracesIndicesRespons4, _tracesIndicesRespons5, _tracesIndicesRespons6, _tracesIndicesRespons7, _tracesIndicesRespons8, _tracesIndicesRespons9, _tracesIndicesRespons10, _response$_shards$tot2, _response$_shards2, _response$_all$total$, _response$_all3, _response$_all3$total, _response$_all3$total2, _response$_all$total$2, _response$_all4, _response$_all4$total, _response$_all4$total2;
    const dataStreamStatsDictionary = {};
    const metricSetsSupportingRollUps = ['service_destination', 'service_transaction', 'service_summary', 'transaction', 'span_breakdown'];
    const metricSetsNotSupportingRollUps = ['app'];
    const rollUpIntervals = [_rollup.RollupInterval.OneMinute, _rollup.RollupInterval.TenMinutes, _rollup.RollupInterval.SixtyMinutes];
    const populateDataStreamStatsDict = (ds, key, response) => {
      var _response$_shards$tot, _response$_shards, _response$_all$primar, _response$_all, _response$_all$primar2, _response$_all$primar3, _response$_all$primar4, _response$_all2, _response$_all2$prima, _response$_all2$prima2;
      ds[key] = ds[key] || {};
      ds[key].all = {
        total: {
          shards: (_response$_shards$tot = response === null || response === void 0 ? void 0 : (_response$_shards = response._shards) === null || _response$_shards === void 0 ? void 0 : _response$_shards.total) !== null && _response$_shards$tot !== void 0 ? _response$_shards$tot : 0,
          docs: {
            count: (_response$_all$primar = response === null || response === void 0 ? void 0 : (_response$_all = response._all) === null || _response$_all === void 0 ? void 0 : (_response$_all$primar2 = _response$_all.primaries) === null || _response$_all$primar2 === void 0 ? void 0 : (_response$_all$primar3 = _response$_all$primar2.docs) === null || _response$_all$primar3 === void 0 ? void 0 : _response$_all$primar3.count) !== null && _response$_all$primar !== void 0 ? _response$_all$primar : 0
          },
          store: {
            size_in_bytes: (_response$_all$primar4 = response === null || response === void 0 ? void 0 : (_response$_all2 = response._all) === null || _response$_all2 === void 0 ? void 0 : (_response$_all2$prima = _response$_all2.primaries) === null || _response$_all2$prima === void 0 ? void 0 : (_response$_all2$prima2 = _response$_all2$prima.store) === null || _response$_all2$prima2 === void 0 ? void 0 : _response$_all2$prima2.size_in_bytes) !== null && _response$_all$primar4 !== void 0 ? _response$_all$primar4 : 0
          }
        }
      };
      return ds;
    };

    // The API calls must be done in series rather than in parallel due to the nature
    // of how tasks are executed. We don't want to burden the customers instances
    // which could directly impact the performance on the UI.
    const fetchRollupMetrics = async () => {
      for (const metricSet of metricSetsSupportingRollUps) {
        for (const bucketSize of rollUpIntervals) {
          const datastream = `metrics-apm.${metricSet}.${bucketSize}-*`;
          const response = await telemetryClient.indicesStats({
            index: [datastream],
            expand_wildcards: 'all',
            filter_path: ['_all.primaries.docs', '_all.primaries.store', '_shards']
          });
          populateDataStreamStatsDict(dataStreamStatsDictionary, `${metricSet}-${bucketSize}`, response);
        }
      }
    };
    const fetchMetricWithoutRollup = async () => {
      for (const metricSet of metricSetsNotSupportingRollUps) {
        const datastream = `metrics-apm.${metricSet}*`;
        const response = await telemetryClient.indicesStats({
          index: [datastream],
          expand_wildcards: 'all',
          filter_path: ['_all.primaries.docs', '_all.primaries.store', '_shards']
        });
        populateDataStreamStatsDict(dataStreamStatsDictionary, metricSet, response);
      }
    };
    await fetchRollupMetrics();
    await fetchMetricWithoutRollup();
    const lastDayStatsResponse = await telemetryClient.search({
      index: [indices.metric],
      expand_wildcards: 'all',
      track_total_hits: false,
      size: 0,
      timeout,
      query: range1d,
      aggs: {
        metricsets: {
          terms: {
            field: _apm.METRICSET_NAME
          },
          aggs: {
            rollup_interval: {
              terms: {
                field: _apm.METRICSET_INTERVAL
              },
              aggs: {
                metrics_value_count: {
                  value_count: {
                    field: _apm.METRICSET_INTERVAL
                  }
                }
              }
            }
          }
        }
      }
    });
    for (const metricSet of metricSetsSupportingRollUps) {
      var _lastDayStatsResponse, _lastDayStatsResponse2, _lastDayStatsResponse3, _metricSetData$rollup;
      const metricSetData = (_lastDayStatsResponse = lastDayStatsResponse.aggregations) === null || _lastDayStatsResponse === void 0 ? void 0 : (_lastDayStatsResponse2 = _lastDayStatsResponse.metricsets) === null || _lastDayStatsResponse2 === void 0 ? void 0 : (_lastDayStatsResponse3 = _lastDayStatsResponse2.buckets) === null || _lastDayStatsResponse3 === void 0 ? void 0 : _lastDayStatsResponse3.find(bucket => bucket.key === metricSet);
      rollUpIntervals.forEach(interval => {
        const key = `${metricSet}-${interval}`;
        dataStreamStatsDictionary[key]['1d'] = {
          doc_count: 0
        };
      });
      if (metricSetData !== null && metricSetData !== void 0 && (_metricSetData$rollup = metricSetData.rollup_interval) !== null && _metricSetData$rollup !== void 0 && _metricSetData$rollup.buckets) {
        for (const intervalBucket of metricSetData.rollup_interval.buckets) {
          const intervalKey = intervalBucket.key;
          const intervalDocCount = intervalBucket.metrics_value_count.value;
          dataStreamStatsDictionary[`${metricSet}-${intervalKey}`]['1d'] = {
            doc_count: intervalDocCount
          };
        }
      }
    }
    for (const metricSet of metricSetsNotSupportingRollUps) {
      var _lastDayStatsResponse4, _lastDayStatsResponse5, _lastDayStatsResponse6;
      const metricSetData = (_lastDayStatsResponse4 = lastDayStatsResponse.aggregations) === null || _lastDayStatsResponse4 === void 0 ? void 0 : (_lastDayStatsResponse5 = _lastDayStatsResponse4.metricsets) === null || _lastDayStatsResponse5 === void 0 ? void 0 : (_lastDayStatsResponse6 = _lastDayStatsResponse5.buckets) === null || _lastDayStatsResponse6 === void 0 ? void 0 : _lastDayStatsResponse6.find(bucket => bucket.key === metricSet);
      dataStreamStatsDictionary[metricSet]['1d'] = {
        doc_count: (metricSetData === null || metricSetData === void 0 ? void 0 : metricSetData.doc_count) || 0
      };
    }
    const response = await telemetryClient.indicesStats({
      index: [_server.APM_AGENT_CONFIGURATION_INDEX, indices.error, indices.metric, indices.onboarding, indices.span, indices.transaction]
    });
    const metricIndicesResponse = await telemetryClient.indicesStats({
      index: [indices.metric]
    });
    const tracesIndicesResponse = await telemetryClient.indicesStats({
      index: [indices.span, indices.transaction]
    });
    return {
      indices: {
        metric: {
          shards: {
            total: (_metricIndicesRespons = (_metricIndicesRespons2 = metricIndicesResponse._shards) === null || _metricIndicesRespons2 === void 0 ? void 0 : _metricIndicesRespons2.total) !== null && _metricIndicesRespons !== void 0 ? _metricIndicesRespons : 0
          },
          all: {
            total: {
              docs: {
                count: (_metricIndicesRespons3 = (_metricIndicesRespons4 = metricIndicesResponse._all) === null || _metricIndicesRespons4 === void 0 ? void 0 : (_metricIndicesRespons5 = _metricIndicesRespons4.total) === null || _metricIndicesRespons5 === void 0 ? void 0 : (_metricIndicesRespons6 = _metricIndicesRespons5.docs) === null || _metricIndicesRespons6 === void 0 ? void 0 : _metricIndicesRespons6.count) !== null && _metricIndicesRespons3 !== void 0 ? _metricIndicesRespons3 : 0
              },
              store: {
                size_in_bytes: (_metricIndicesRespons7 = (_metricIndicesRespons8 = metricIndicesResponse._all) === null || _metricIndicesRespons8 === void 0 ? void 0 : (_metricIndicesRespons9 = _metricIndicesRespons8.total) === null || _metricIndicesRespons9 === void 0 ? void 0 : (_metricIndicesRespons10 = _metricIndicesRespons9.store) === null || _metricIndicesRespons10 === void 0 ? void 0 : _metricIndicesRespons10.size_in_bytes) !== null && _metricIndicesRespons7 !== void 0 ? _metricIndicesRespons7 : 0
              }
            }
          },
          metricset: {
            ...dataStreamStatsDictionary
          }
        },
        traces: {
          shards: {
            total: (_tracesIndicesRespons = (_tracesIndicesRespons2 = tracesIndicesResponse._shards) === null || _tracesIndicesRespons2 === void 0 ? void 0 : _tracesIndicesRespons2.total) !== null && _tracesIndicesRespons !== void 0 ? _tracesIndicesRespons : 0
          },
          all: {
            total: {
              docs: {
                count: (_tracesIndicesRespons3 = (_tracesIndicesRespons4 = tracesIndicesResponse._all) === null || _tracesIndicesRespons4 === void 0 ? void 0 : (_tracesIndicesRespons5 = _tracesIndicesRespons4.total) === null || _tracesIndicesRespons5 === void 0 ? void 0 : (_tracesIndicesRespons6 = _tracesIndicesRespons5.docs) === null || _tracesIndicesRespons6 === void 0 ? void 0 : _tracesIndicesRespons6.count) !== null && _tracesIndicesRespons3 !== void 0 ? _tracesIndicesRespons3 : 0
              },
              store: {
                size_in_bytes: (_tracesIndicesRespons7 = (_tracesIndicesRespons8 = tracesIndicesResponse._all) === null || _tracesIndicesRespons8 === void 0 ? void 0 : (_tracesIndicesRespons9 = _tracesIndicesRespons8.total) === null || _tracesIndicesRespons9 === void 0 ? void 0 : (_tracesIndicesRespons10 = _tracesIndicesRespons9.store) === null || _tracesIndicesRespons10 === void 0 ? void 0 : _tracesIndicesRespons10.size_in_bytes) !== null && _tracesIndicesRespons7 !== void 0 ? _tracesIndicesRespons7 : 0
              }
            }
          }
        },
        shards: {
          total: (_response$_shards$tot2 = (_response$_shards2 = response._shards) === null || _response$_shards2 === void 0 ? void 0 : _response$_shards2.total) !== null && _response$_shards$tot2 !== void 0 ? _response$_shards$tot2 : 0
        },
        all: {
          total: {
            docs: {
              count: (_response$_all$total$ = (_response$_all3 = response._all) === null || _response$_all3 === void 0 ? void 0 : (_response$_all3$total = _response$_all3.total) === null || _response$_all3$total === void 0 ? void 0 : (_response$_all3$total2 = _response$_all3$total.docs) === null || _response$_all3$total2 === void 0 ? void 0 : _response$_all3$total2.count) !== null && _response$_all$total$ !== void 0 ? _response$_all$total$ : 0
            },
            store: {
              size_in_bytes: (_response$_all$total$2 = (_response$_all4 = response._all) === null || _response$_all4 === void 0 ? void 0 : (_response$_all4$total = _response$_all4.total) === null || _response$_all4$total === void 0 ? void 0 : (_response$_all4$total2 = _response$_all4$total.store) === null || _response$_all4$total2 === void 0 ? void 0 : _response$_all4$total2.size_in_bytes) !== null && _response$_all$total$2 !== void 0 ? _response$_all$total$2 : 0
            }
          }
        }
      }
    };
  }
}, {
  name: 'cardinality',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _rumAgentCardinalityR, _allAgentsCardinality, _rumAgentCardinalityR2, _allAgentsCardinality2, _rumAgentCardinalityR3;
    const allAgentsCardinalityResponse = await telemetryClient.search({
      index: [indices.transaction],
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [range1d]
        }
      },
      aggs: {
        [_apm.TRANSACTION_NAME]: {
          cardinality: {
            field: _apm.TRANSACTION_NAME
          }
        },
        [_apm.USER_AGENT_ORIGINAL]: {
          cardinality: {
            field: _apm.USER_AGENT_ORIGINAL
          }
        }
      }
    });
    const rumAgentCardinalityResponse = await telemetryClient.search({
      index: [indices.transaction],
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [range1d, {
            terms: {
              [_apm.AGENT_NAME]: _agent_names.RUM_AGENT_NAMES
            }
          }]
        }
      },
      aggs: {
        [_apm.CLIENT_GEO_COUNTRY_ISO_CODE]: {
          cardinality: {
            field: _apm.CLIENT_GEO_COUNTRY_ISO_CODE
          }
        },
        [_apm.TRANSACTION_NAME]: {
          cardinality: {
            field: _apm.TRANSACTION_NAME
          }
        },
        [_apm.USER_AGENT_ORIGINAL]: {
          cardinality: {
            field: _apm.USER_AGENT_ORIGINAL
          }
        }
      }
    });
    return {
      cardinality: {
        client: {
          geo: {
            country_iso_code: {
              rum: {
                '1d': (_rumAgentCardinalityR = rumAgentCardinalityResponse.aggregations) === null || _rumAgentCardinalityR === void 0 ? void 0 : _rumAgentCardinalityR[_apm.CLIENT_GEO_COUNTRY_ISO_CODE].value
              }
            }
          }
        },
        transaction: {
          name: {
            all_agents: {
              '1d': (_allAgentsCardinality = allAgentsCardinalityResponse.aggregations) === null || _allAgentsCardinality === void 0 ? void 0 : _allAgentsCardinality[_apm.TRANSACTION_NAME].value
            },
            rum: {
              '1d': (_rumAgentCardinalityR2 = rumAgentCardinalityResponse.aggregations) === null || _rumAgentCardinalityR2 === void 0 ? void 0 : _rumAgentCardinalityR2[_apm.TRANSACTION_NAME].value
            }
          }
        },
        user_agent: {
          original: {
            all_agents: {
              '1d': (_allAgentsCardinality2 = allAgentsCardinalityResponse.aggregations) === null || _allAgentsCardinality2 === void 0 ? void 0 : _allAgentsCardinality2[_apm.USER_AGENT_ORIGINAL].value
            },
            rum: {
              '1d': (_rumAgentCardinalityR3 = rumAgentCardinalityResponse.aggregations) === null || _rumAgentCardinalityR3 === void 0 ? void 0 : _rumAgentCardinalityR3[_apm.USER_AGENT_ORIGINAL].value
            }
          }
        }
      }
    };
  }
}, {
  name: 'service_groups',
  executor: async ({
    savedObjectsClient
  }) => {
    var _response$total;
    const response = await savedObjectsClient.find({
      type: _service_groups.APM_SERVICE_GROUP_SAVED_OBJECT_TYPE,
      page: 1,
      perPage: _service_groups.MAX_NUMBER_OF_SERVICE_GROUPS,
      sortField: 'updated_at',
      sortOrder: 'desc',
      namespaces: ['*']
    });
    const kueryExpressions = response.saved_objects.map(({
      attributes: {
        kuery
      }
    }) => kuery);
    const kueryFields = kueryExpressions.map(_esQuery.getKqlFieldNamesFromExpression).reduce((a, b) => a.concat(b), []);
    return {
      service_groups: {
        kuery_fields: (0, _lodash.uniq)(kueryFields),
        total: (_response$total = response.total) !== null && _response$total !== void 0 ? _response$total : 0
      }
    };
  }
}, {
  name: 'custom_dashboards',
  executor: async ({
    savedObjectsClient
  }) => {
    var _response$total2;
    const response = await savedObjectsClient.find({
      type: _custom_dashboards.APM_CUSTOM_DASHBOARDS_SAVED_OBJECT_TYPE,
      page: 1,
      perPage: 500,
      sortField: 'updated_at',
      sortOrder: 'desc',
      namespaces: ['*']
    });
    const kueryExpressions = response.saved_objects.map(({
      attributes: {
        kuery
      }
    }) => kuery !== null && kuery !== void 0 ? kuery : '');
    const kueryFields = kueryExpressions.map(_esQuery.getKqlFieldNamesFromExpression).reduce((a, b) => a.concat(b), []);
    return {
      custom_dashboards: {
        kuery_fields: (0, _lodash.uniq)(kueryFields),
        total: (_response$total2 = response.total) !== null && _response$total2 !== void 0 ? _response$total2 : 0
      }
    };
  }
}, {
  name: 'per_service',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$aggregation8, _response$aggregation9;
    const response = await telemetryClient.search({
      index: [indices.transaction],
      track_total_hits: false,
      size: 0,
      timeout,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: 'now-1h'
              }
            }
          }, {
            term: {
              [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
            }
          }]
        }
      },
      aggs: {
        service_names: {
          terms: {
            field: _apm.SERVICE_NAME,
            size: 2500
          },
          aggs: {
            environments: {
              terms: {
                field: _apm.SERVICE_ENVIRONMENT,
                size: 5
              },
              aggs: {
                instances: {
                  cardinality: {
                    field: _apm.SERVICE_NODE_NAME
                  }
                },
                transaction_types: {
                  cardinality: {
                    field: _apm.TRANSACTION_TYPE
                  }
                },
                top_metrics: {
                  top_metrics: {
                    sort: '_score',
                    metrics: [{
                      field: _apm.AGENT_ACTIVATION_METHOD
                    }, {
                      field: _apm.AGENT_NAME
                    }, {
                      field: _apm.AGENT_VERSION
                    }, {
                      field: _apm.SERVICE_LANGUAGE_NAME
                    }, {
                      field: _apm.SERVICE_LANGUAGE_VERSION
                    }, {
                      field: _apm.SERVICE_FRAMEWORK_NAME
                    }, {
                      field: _apm.SERVICE_FRAMEWORK_VERSION
                    }, {
                      field: _apm.SERVICE_RUNTIME_NAME
                    }, {
                      field: _apm.SERVICE_RUNTIME_VERSION
                    }, {
                      field: _apm.KUBERNETES_POD_NAME
                    }, {
                      field: _apm.CONTAINER_ID
                    }]
                  }
                },
                [_apm.CLOUD_REGION]: {
                  terms: {
                    field: _apm.CLOUD_REGION,
                    size: 5
                  }
                },
                [_apm.CLOUD_PROVIDER]: {
                  terms: {
                    field: _apm.CLOUD_PROVIDER,
                    size: 3
                  }
                },
                [_apm.CLOUD_AVAILABILITY_ZONE]: {
                  terms: {
                    field: _apm.CLOUD_AVAILABILITY_ZONE,
                    size: 5
                  }
                },
                [_apm.FAAS_TRIGGER_TYPE]: {
                  terms: {
                    field: _apm.FAAS_TRIGGER_TYPE,
                    size: 5
                  }
                }
              }
            }
          }
        }
      }
    });
    const serviceBuckets = (_response$aggregation8 = (_response$aggregation9 = response.aggregations) === null || _response$aggregation9 === void 0 ? void 0 : _response$aggregation9.service_names.buckets) !== null && _response$aggregation8 !== void 0 ? _response$aggregation8 : [];
    const data = serviceBuckets.flatMap(serviceBucket => {
      var _serviceBucket$enviro, _serviceBucket$enviro2;
      const envHash = (0, _crypto.createHash)('sha256').update(serviceBucket.key).digest('hex');
      const envBuckets = (_serviceBucket$enviro = (_serviceBucket$enviro2 = serviceBucket.environments) === null || _serviceBucket$enviro2 === void 0 ? void 0 : _serviceBucket$enviro2.buckets) !== null && _serviceBucket$enviro !== void 0 ? _serviceBucket$enviro : [];
      return envBuckets.map(envBucket => {
        var _envBucket$instances$, _envBucket$transactio, _envBucket$CLOUD_AVAI, _envBucket$CLOUD_AVAI2, _envBucket$CLOUD_REGI, _envBucket$CLOUD_REGI2, _envBucket$CLOUD_PROV, _envBucket$CLOUD_PROV2, _envBucket$FAAS_TRIGG, _envBucket$FAAS_TRIGG2, _envBucket$top_metric, _envBucket$top_metric2, _envBucket$top_metric3, _envBucket$top_metric4, _envBucket$top_metric5, _envBucket$top_metric6, _envBucket$top_metric7, _envBucket$top_metric8, _envBucket$top_metric9, _envBucket$top_metric10, _envBucket$top_metric11;
        const nameHash = (0, _crypto.createHash)('sha256').update(envBucket.key).digest('hex');
        const fullServiceName = `${nameHash}~${envHash}`;
        return {
          service_id: fullServiceName,
          timed_out: response.timed_out,
          num_service_nodes: (_envBucket$instances$ = envBucket.instances.value) !== null && _envBucket$instances$ !== void 0 ? _envBucket$instances$ : 1,
          num_transaction_types: (_envBucket$transactio = envBucket.transaction_types.value) !== null && _envBucket$transactio !== void 0 ? _envBucket$transactio : 0,
          cloud: {
            availability_zones: (_envBucket$CLOUD_AVAI = (_envBucket$CLOUD_AVAI2 = envBucket[_apm.CLOUD_AVAILABILITY_ZONE]) === null || _envBucket$CLOUD_AVAI2 === void 0 ? void 0 : _envBucket$CLOUD_AVAI2.buckets.map(inner => inner.key)) !== null && _envBucket$CLOUD_AVAI !== void 0 ? _envBucket$CLOUD_AVAI : [],
            regions: (_envBucket$CLOUD_REGI = (_envBucket$CLOUD_REGI2 = envBucket[_apm.CLOUD_REGION]) === null || _envBucket$CLOUD_REGI2 === void 0 ? void 0 : _envBucket$CLOUD_REGI2.buckets.map(inner => inner.key)) !== null && _envBucket$CLOUD_REGI !== void 0 ? _envBucket$CLOUD_REGI : [],
            providers: (_envBucket$CLOUD_PROV = (_envBucket$CLOUD_PROV2 = envBucket[_apm.CLOUD_PROVIDER]) === null || _envBucket$CLOUD_PROV2 === void 0 ? void 0 : _envBucket$CLOUD_PROV2.buckets.map(inner => inner.key)) !== null && _envBucket$CLOUD_PROV !== void 0 ? _envBucket$CLOUD_PROV : []
          },
          faas: {
            trigger: {
              type: (_envBucket$FAAS_TRIGG = (_envBucket$FAAS_TRIGG2 = envBucket[_apm.FAAS_TRIGGER_TYPE]) === null || _envBucket$FAAS_TRIGG2 === void 0 ? void 0 : _envBucket$FAAS_TRIGG2.buckets.map(inner => inner.key)) !== null && _envBucket$FAAS_TRIGG !== void 0 ? _envBucket$FAAS_TRIGG : []
            }
          },
          agent: {
            name: (_envBucket$top_metric = envBucket.top_metrics) === null || _envBucket$top_metric === void 0 ? void 0 : _envBucket$top_metric.top[0].metrics[_apm.AGENT_NAME],
            activation_method: (_envBucket$top_metric2 = envBucket.top_metrics) === null || _envBucket$top_metric2 === void 0 ? void 0 : _envBucket$top_metric2.top[0].metrics[_apm.AGENT_ACTIVATION_METHOD],
            version: (_envBucket$top_metric3 = envBucket.top_metrics) === null || _envBucket$top_metric3 === void 0 ? void 0 : _envBucket$top_metric3.top[0].metrics[_apm.AGENT_VERSION]
          },
          service: {
            language: {
              name: (_envBucket$top_metric4 = envBucket.top_metrics) === null || _envBucket$top_metric4 === void 0 ? void 0 : _envBucket$top_metric4.top[0].metrics[_apm.SERVICE_LANGUAGE_NAME],
              version: (_envBucket$top_metric5 = envBucket.top_metrics) === null || _envBucket$top_metric5 === void 0 ? void 0 : _envBucket$top_metric5.top[0].metrics[_apm.SERVICE_LANGUAGE_VERSION]
            },
            framework: {
              name: (_envBucket$top_metric6 = envBucket.top_metrics) === null || _envBucket$top_metric6 === void 0 ? void 0 : _envBucket$top_metric6.top[0].metrics[_apm.SERVICE_FRAMEWORK_NAME],
              version: (_envBucket$top_metric7 = envBucket.top_metrics) === null || _envBucket$top_metric7 === void 0 ? void 0 : _envBucket$top_metric7.top[0].metrics[_apm.SERVICE_FRAMEWORK_VERSION]
            },
            runtime: {
              name: (_envBucket$top_metric8 = envBucket.top_metrics) === null || _envBucket$top_metric8 === void 0 ? void 0 : _envBucket$top_metric8.top[0].metrics[_apm.SERVICE_RUNTIME_NAME],
              version: (_envBucket$top_metric9 = envBucket.top_metrics) === null || _envBucket$top_metric9 === void 0 ? void 0 : _envBucket$top_metric9.top[0].metrics[_apm.SERVICE_RUNTIME_VERSION]
            }
          },
          kubernetes: {
            pod: {
              name: (_envBucket$top_metric10 = envBucket.top_metrics) === null || _envBucket$top_metric10 === void 0 ? void 0 : _envBucket$top_metric10.top[0].metrics[_apm.KUBERNETES_POD_NAME]
            }
          },
          container: {
            id: (_envBucket$top_metric11 = envBucket.top_metrics) === null || _envBucket$top_metric11 === void 0 ? void 0 : _envBucket$top_metric11.top[0].metrics[_apm.CONTAINER_ID]
          }
        };
      });
    });
    return {
      per_service: data
    };
  }
}, {
  name: 'top_traces',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$aggregation10, _response$aggregation11, _response$aggregation12, _response$aggregation13;
    const response = await telemetryClient.search({
      index: [indices.transaction, indices.span, indices.error],
      size: 0,
      track_total_hits: false,
      timeout,
      query: {
        bool: {
          filter: [range1d]
        }
      },
      aggs: {
        top_traces: {
          terms: {
            field: _apm.TRACE_ID,
            size: 100
          }
        },
        max: {
          max_bucket: {
            buckets_path: 'top_traces>_count'
          }
        },
        median: {
          percentiles_bucket: {
            buckets_path: 'top_traces>_count',
            percents: [50]
          }
        }
      }
    });
    return {
      top_traces: {
        max: (_response$aggregation10 = (_response$aggregation11 = response.aggregations) === null || _response$aggregation11 === void 0 ? void 0 : _response$aggregation11.max.value) !== null && _response$aggregation10 !== void 0 ? _response$aggregation10 : 0,
        median: (_response$aggregation12 = (_response$aggregation13 = response.aggregations) === null || _response$aggregation13 === void 0 ? void 0 : _response$aggregation13.median.values['50.0']) !== null && _response$aggregation12 !== void 0 ? _response$aggregation12 : 0
      }
    };
  }
}];