"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createWorkflowRunner = void 0;
var _uuid = require("uuid");
var _wcFrameworkTypesServer = require("@kbn/wc-framework-types-server");
var _scoped_runner = require("./scoped_runner");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createWorkflowRunner = params => {
  const {
    workflowRegistry,
    nodeRegistry,
    modelProviderFactory,
    toolRegistry,
    logger,
    core: {
      elasticsearch
    }
  } = params;
  const getWorkflowDefinition = async workflowId => {
    if (workflowRegistry.has(workflowId)) {
      return workflowRegistry.get(workflowId);
    }
    return undefined;
  };
  const run = async options => {
    const {
      id: workflowId,
      request,
      inputs,
      onEvent,
      defaultConnectorId,
      toolProvider: customToolProvider
    } = options;
    const runId = (0, _uuid.v4)();
    const executionState = (0, _utils.createInitialExecutionState)({
      runId,
      workflowId
    });
    const workflowDefinition = await getWorkflowDefinition(workflowId);
    if (!workflowDefinition) {
      throw new _wcFrameworkTypesServer.WorkflowExecutionError(`Workflow with id [${workflowId}] not found in registry`, 'workflowNotFound', {
        state: executionState
      });
    }
    const modelProvider = await modelProviderFactory({
      request,
      defaultConnectorId
    });
    const internalContext = {
      logger,
      request,
      modelProvider,
      workflowRegistry,
      nodeRegistry,
      toolProvider: customToolProvider ? (0, _utils.combineToolProviders)(customToolProvider, toolRegistry.asToolProvider()) : toolRegistry.asToolProvider(),
      esClusterClient: elasticsearch.client.asScoped(request),
      // TODO: we probably need to always have a default one, for internal telemetry, and combine with onEvent if present
      eventHandler: onEvent !== null && onEvent !== void 0 ? onEvent : () => undefined,
      executionState
    };
    const scopedRunner = (0, _scoped_runner.createInternalRunner)({
      internalContext
    });
    return scopedRunner.runWorkflow({
      workflowDefinition,
      inputs
    });
  };
  return {
    run
  };
};
exports.createWorkflowRunner = createWorkflowRunner;