"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchAgentGraph = void 0;
var _zod = require("@kbn/zod");
var _langgraph = require("@langchain/langgraph");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _tool_messages = require("./utils/tool_messages");
var _prompts = require("./prompts");
var _workflow_tools = require("./workflow_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSearchAgentGraph = async ({
  chatModel,
  toolsProvider,
  logger
}) => {
  const StateAnnotation = _langgraph.Annotation.Root({
    // input
    searchQuery: _langgraph.Annotation,
    searchContext: _langgraph.Annotation,
    // internal state
    planning: _langgraph.Annotation,
    retrievalMessages: (0, _langgraph.Annotation)({
      reducer: _langgraph.messagesStateReducer,
      default: () => []
    }),
    ratings: _langgraph.Annotation,
    parsedResults: _langgraph.Annotation,
    filteredResults: _langgraph.Annotation,
    // output
    citations: _langgraph.Annotation,
    summary: _langgraph.Annotation
  });
  const searchTools = await toolsProvider.getSearchTools();
  const retrievalTools = [...searchTools, (0, _workflow_tools.stepDoneTool)()];

  // step one - create retrieval plan

  const planningModel = chatModel.bindTools(retrievalTools).withConfig({
    tags: [`agent:search_agent`, `step:planning`]
  });
  const createPlanning = async state => {
    logger.debug('Creating a plan');
    const response = await planningModel.invoke((0, _prompts.getPlanningPrompt)({
      query: state.searchQuery
    }));
    return {
      planning: response.content
    };
  };

  // step two - apply the plan by executing tool calls

  const retrievalToolNode = new _prebuilt.ToolNode(retrievalTools);
  const retrievalModel = chatModel.bindTools(retrievalTools).withConfig({
    tags: [`agent:search_agent`, `step:retrieval`]
  });
  const agenticRetrieval = async state => {
    logger.debug(`Running retrieval with plan: ${state.planning}`);
    const response = await retrievalModel.invoke((0, _prompts.getRetrievalPrompt)({
      plan: state.planning,
      messages: state.retrievalMessages
    }));
    return {
      retrievalMessages: [response]
    };
  };
  const handleRetrievalTransition = async state => {
    var _lastMessage$tool_cal;
    const messages = state.retrievalMessages;
    const lastMessage = messages[messages.length - 1];
    if (lastMessage && (_lastMessage$tool_cal = lastMessage.tool_calls) !== null && _lastMessage$tool_cal !== void 0 && _lastMessage$tool_cal.length) {
      const toolNames = lastMessage.tool_calls.map(call => call.name);
      if (toolNames.length === 1 && toolNames[0] === _workflow_tools.stepDoneToolName) {
        logger.debug('Transitioning to process results');
        return 'process_results';
      } else {
        logger.debug('Transitioning to call retrieval tools');
        return 'call_retrieval_tools';
      }
    }
    // should not happen, LLM is supposed to always call tools at each round, but we never know
    return 'process_results';
  };
  const retrievalToolHandler = async state => {
    logger.debug('Handling retrieval outputs');
    const toolNodeResult = await retrievalToolNode.invoke(state.retrievalMessages);
    return {
      retrievalMessages: [...toolNodeResult]
    };
  };

  // step 3: rate documents

  const processResults = async state => {
    logger.debug('Processing results');
    const toolResults = (0, _tool_messages.processSearchResults)(state.retrievalMessages);
    return {
      parsedResults: toolResults.results
    };
  };
  const analysisModel = chatModel.withStructuredOutput(_zod.z.object({
    ratings: _zod.z.array(_zod.z.string()).describe('the ratings, one per document using the "{id}|{grade}" format.'),
    comment: _zod.z.string().optional().describe('optional comments or remarks on the ratings')
  })).withConfig({
    tags: [`agent:search_agent`, `step:analysis`]
  });
  const rateResults = async state => {
    logger.debug(`Rating ${state.parsedResults.length} results`);
    const results = state.parsedResults;
    const query = state.searchQuery;
    const response = await analysisModel.invoke((0, _prompts.getAnalysisPrompt)({
      query,
      results
    }));
    const ratings = (0, _tool_messages.parseRatings)(response.ratings);
    const filteredResults = (0, _tool_messages.processRatings)({
      results,
      ratings,
      maxResults: 5,
      minScore: 5
    });
    const citations = filteredResults.map(result => result.reference);
    return {
      ratings,
      filteredResults,
      citations
    };
  };

  // step 4: create summary

  const summarizerModel = chatModel.bindTools(retrievalTools).withConfig({
    tags: [`agent:search_agent`, `step:summary`]
  });
  const generateSummary = async state => {
    logger.debug(`Generating a summary for ${state.searchQuery}`);
    const response = await summarizerModel.invoke((0, _prompts.getSummarizerPrompt)({
      query: state.searchQuery,
      context: state.searchContext,
      results: state.filteredResults
    }));
    return {
      summary: response.content
    };
  };

  // step 5: end

  const graph = new _langgraph.StateGraph(StateAnnotation)
  // nodes
  .addNode('create_planning', createPlanning).addNode('retrieve_agent', agenticRetrieval).addNode('call_retrieval_tools', retrievalToolHandler).addNode('process_results', processResults).addNode('rate_results', rateResults).addNode('generate_summary', generateSummary)
  // edges
  .addEdge('__start__', 'create_planning').addEdge('create_planning', 'retrieve_agent').addConditionalEdges('retrieve_agent', handleRetrievalTransition, {
    call_retrieval_tools: 'call_retrieval_tools',
    process_results: 'process_results'
  }).addEdge('call_retrieval_tools', 'retrieve_agent').addEdge('process_results', 'rate_results').addEdge('rate_results', 'generate_summary').addEdge('generate_summary', '__end__')
  // end
  .compile();
  return graph;
};
exports.createSearchAgentGraph = createSearchAgentGraph;