"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAgentRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerAgentRoutes = ({
  getServices,
  router,
  logger
}) => {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // API to get a single agent
  router.get({
    path: '/internal/workchat/agents/{agentId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.useWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        agentId: _configSchema.schema.string()
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      agentService
    } = getServices();
    const client = await agentService.getScopedClient({
      request
    });
    const {
      agentId
    } = request.params;
    const agent = await client.get({
      agentId
    });
    return res.ok({
      body: agent
    });
  }));

  // API to create an agent
  router.post({
    path: '/internal/workchat/agents',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string({
          defaultValue: ''
        }),
        configuration: _configSchema.schema.object({
          systemPrompt: _configSchema.schema.maybe(_configSchema.schema.string()),
          useCase: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        public: _configSchema.schema.boolean({
          defaultValue: false
        }),
        avatar: _configSchema.schema.object({
          color: _configSchema.schema.maybe(_configSchema.schema.string()),
          text: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const payload = request.body;
    const {
      agentService
    } = getServices();
    const client = await agentService.getScopedClient({
      request
    });

    // TODO: validation

    const agent = await client.create(payload);
    return res.ok({
      body: {
        success: true,
        agent
      }
    });
  }));

  // API to update an agent
  router.put({
    path: '/internal/workchat/agents/{agentId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        agentId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string({
          defaultValue: ''
        }),
        configuration: _configSchema.schema.object({
          systemPrompt: _configSchema.schema.maybe(_configSchema.schema.string()),
          useCase: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        avatar: _configSchema.schema.object({
          color: _configSchema.schema.maybe(_configSchema.schema.string()),
          text: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        public: _configSchema.schema.boolean({
          defaultValue: false
        })
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      agentId
    } = request.params;
    const payload = request.body;
    const {
      agentService
    } = getServices();
    const client = await agentService.getScopedClient({
      request
    });

    // TODO: validation

    const agent = await client.update(agentId, payload);
    return res.ok({
      body: {
        success: true,
        agent
      }
    });
  }));
  router.delete({
    path: '/internal/workchat/agents/{agentId}',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.manageWorkchat]
      }
    },
    validate: {
      params: _configSchema.schema.object({
        agentId: _configSchema.schema.string()
      })
    }
  }, wrapHandler(async (ctx, request, res) => {
    const {
      agentId
    } = request.params;
    const {
      agentService
    } = getServices();
    const client = await agentService.getScopedClient({
      request
    });
    const didDelete = await client.delete(agentId);
    return res.ok({
      body: {
        success: didDelete
      }
    });
  }));

  // API to list all accessible agents
  router.get({
    path: '/internal/workchat/agents',
    security: {
      authz: {
        requiredPrivileges: [_features.apiCapabilities.useWorkchat]
      }
    },
    validate: false
  }, wrapHandler(async (ctx, request, res) => {
    const {
      agentService
    } = getServices();
    const client = await agentService.getScopedClient({
      request
    });
    const agents = await client.list();
    return res.ok({
      body: {
        agents
      }
    });
  }));
};
exports.registerAgentRoutes = registerAgentRoutes;