"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateRegExpString = exports.getReindexProgressLabel = exports.getDeprecationsUpperLimit = exports.getDataStreamReindexProgressLabel = exports.getDataStreamReindexProgress = void 0;
var _pipeable = require("fp-ts/pipeable");
var _Either = require("fp-ts/Either");
var _constants = require("../../../common/constants");
var _types = require("../../../common/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validateRegExpString = s => (0, _pipeable.pipe)((0, _Either.tryCatch)(() => new RegExp(s), e => e.message), (0, _Either.fold)(errorMessage => errorMessage, () => ''));

/*
 * There isnt much difference between having 1M or 1.1M deprecation warnings, the number is
 * so big it beats the purpose of having a little preview of the count. With this we can also
 * prevent the container of the value to grow due to the value being so large.
 */
exports.validateRegExpString = validateRegExpString;
const getDeprecationsUpperLimit = count => {
  if (count > _constants.DEPRECATION_WARNING_UPPER_LIMIT) {
    return `${_constants.DEPRECATION_WARNING_UPPER_LIMIT}+`;
  }
  return count.toString();
};

/*
 * Reindexing task consists of 5 (or 6) steps: making the index read-only, creating a new index,
 * reindexing documents into the new index, creating an alias to point to the newly created index,
 * delete the original index, (optionally) update existing aliases to point to the new index.
 * Steps 1, 2 and 4, 5 & 6 each contribute 5% to the overall progress.
 * Step 3 (reindexing documents) can take a long time for large indices and its progress is calculated
 * between 10% and 90% of the overall progress depending on its completeness percentage.
 */
exports.getDeprecationsUpperLimit = getDeprecationsUpperLimit;
const getReindexProgressLabel = (reindexTaskPercComplete, lastCompletedStep, hasExistingAliases = false) => {
  let percentsComplete = 0;
  switch (lastCompletedStep) {
    case _types.ReindexStep.created:
      // the reindex task has just started, 0% progress
      percentsComplete = 0;
      break;
    case _types.ReindexStep.readonly:
      {
        // step 1 completed, 5% progress
        percentsComplete = 5;
        break;
      }
    case _types.ReindexStep.newIndexCreated:
      {
        // step 2 completed, 10% progress
        percentsComplete = 10;
        break;
      }
    case _types.ReindexStep.reindexStarted:
      {
        // step 3 started, 10-95% progress depending on progress of reindexing documents in ES
        percentsComplete = reindexTaskPercComplete !== null ? 10 + Math.round(reindexTaskPercComplete * (hasExistingAliases ? 75 : 80)) : 10;
        break;
      }
    case _types.ReindexStep.reindexCompleted:
      {
        // step 3 completed
        percentsComplete = hasExistingAliases ? 85 : 90;
        break;
      }
    case _types.ReindexStep.indexSettingsRestored:
      {
        // step 4 completed
        percentsComplete = hasExistingAliases ? 87 : 92;
        break;
      }
    case _types.ReindexStep.aliasCreated:
      {
        // step 5 completed
        percentsComplete = hasExistingAliases ? 90 : 95;
        break;
      }
    case _types.ReindexStep.originalIndexDeleted:
      {
        // step 6 completed
        percentsComplete = hasExistingAliases ? 95 : 100;
        break;
      }
    case _types.ReindexStep.existingAliasesUpdated:
      {
        // step 7 completed, 100% progress
        percentsComplete = 100;
        break;
      }
  }
  return `${percentsComplete}%`;
};
exports.getReindexProgressLabel = getReindexProgressLabel;
const getDataStreamReindexProgress = (status, taskPercComplete) => {
  switch (status) {
    case _types.DataStreamMigrationStatus.notStarted:
      return 0;
    case _types.DataStreamMigrationStatus.fetchFailed:
    case _types.DataStreamMigrationStatus.failed:
    case _types.DataStreamMigrationStatus.cancelled:
    case _types.DataStreamMigrationStatus.inProgress:
      {
        return taskPercComplete !== null ? Math.round(taskPercComplete * 100) : 0;
      }
    case _types.DataStreamMigrationStatus.completed:
      {
        return 100;
      }
  }
  return 0;
};
exports.getDataStreamReindexProgress = getDataStreamReindexProgress;
const getDataStreamReindexProgressLabel = (status, taskPercComplete) => {
  const percentsComplete = getDataStreamReindexProgress(status, taskPercComplete);
  return `${percentsComplete}%`;
};
exports.getDataStreamReindexProgressLabel = getDataStreamReindexProgressLabel;