"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isThreatEnrichmentFieldOrSubfield = exports.isRuleParametersFieldOrSubfield = exports.isGeoField = exports.getFieldCategory = exports.getDataFromFieldsHits = exports.formatGeoLocation = exports.baseCategoryFields = void 0;
var _fp = require("lodash/fp");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _ecs_field_map = require("@kbn/rule-registry-plugin/common/assets/field_maps/ecs_field_map");
var _technical_rule_field_map = require("@kbn/rule-registry-plugin/common/assets/field_maps/technical_rule_field_map");
var _alertsAsDataUtils = require("@kbn/alerts-as-data-utils");
var _to_array = require("./to_array");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const baseCategoryFields = exports.baseCategoryFields = ['@timestamp', 'labels', 'message', 'tags'];
const nonFlattenedFormatParamsFields = ['related_integrations', 'threat_mapping'];
const getFieldCategory = field => {
  const fieldCategory = field.split('.')[0];
  if (!(0, _fp.isEmpty)(fieldCategory) && baseCategoryFields.includes(fieldCategory)) {
    return 'base';
  }
  return fieldCategory;
};
exports.getFieldCategory = getFieldCategory;
const formatGeoLocation = item => {
  const itemGeo = item.length > 0 ? item[0] : null;
  if (itemGeo != null && !(0, _fp.isEmpty)(itemGeo.coordinates)) {
    try {
      return (0, _to_array.toStringArray)({
        lon: itemGeo.coordinates[0],
        lat: itemGeo.coordinates[1]
      });
    } catch {
      return (0, _to_array.toStringArray)(item);
    }
  }
  return (0, _to_array.toStringArray)(item);
};
exports.formatGeoLocation = formatGeoLocation;
const isGeoField = field => field.includes('geo.location') || field.includes('geoip.location');
exports.isGeoField = isGeoField;
const isRuleParametersFieldOrSubfield = (field, prependField) => ((prependField === null || prependField === void 0 ? void 0 : prependField.includes(_ruleDataUtils.ALERT_RULE_PARAMETERS)) || field === _ruleDataUtils.ALERT_RULE_PARAMETERS) && !nonFlattenedFormatParamsFields.includes(field);
exports.isRuleParametersFieldOrSubfield = isRuleParametersFieldOrSubfield;
const isThreatEnrichmentFieldOrSubfield = (field, prependField) => (prependField === null || prependField === void 0 ? void 0 : prependField.includes(_constants.ENRICHMENT_DESTINATION_PATH)) || field === _constants.ENRICHMENT_DESTINATION_PATH;

// Helper functions
exports.isThreatEnrichmentFieldOrSubfield = isThreatEnrichmentFieldOrSubfield;
const createFieldItem = (fieldCategory, field, values, isObjectArray) => ({
  category: fieldCategory,
  field,
  values,
  originalValue: values,
  isObjectArray
});
const processGeoField = (field, item, fieldCategory) => {
  const formattedLocation = formatGeoLocation(item);
  return createFieldItem(fieldCategory, field, formattedLocation, true);
};
const processSimpleField = (dotField, strArr, isObjectArray, fieldCategory) => createFieldItem(fieldCategory, dotField, strArr, isObjectArray);
const processNestedFields = (item, dotField, fieldCategory, prependDotField) => {
  if (Array.isArray(item)) {
    return item.flatMap(curr => getDataFromFieldsHits(curr, prependDotField ? dotField : undefined, fieldCategory));
  }
  return getDataFromFieldsHits(item, prependDotField ? dotField : undefined, fieldCategory);
};
// Memoized field maps
const fieldMaps = {
  ..._technical_rule_field_map.technicalRuleFieldMap,
  ..._ecs_field_map.ecsFieldMap,
  ..._alertsAsDataUtils.legacyExperimentalFieldMap
};
const getDataFromFieldsHits = (fields, prependField, prependFieldCategory) => {
  const resultMap = new Map();
  const fieldNames = Object.keys(fields);
  for (let i = 0; i < fieldNames.length; i++) {
    const field = fieldNames[i];
    const item = fields[field];
    const fieldCategory = prependFieldCategory !== null && prependFieldCategory !== void 0 ? prependFieldCategory : getFieldCategory(field);
    const dotField = prependField ? `${prependField}.${field}` : field;

    // Handle geo fields
    if (isGeoField(field)) {
      const geoItem = processGeoField(field, item, fieldCategory);
      resultMap.set(field, geoItem);
      // eslint-disable-next-line no-continue
      continue;
    }
    const objArrStr = (0, _to_array.toObjectArrayOfStrings)(item);
    const strArr = objArrStr.map(({
      str
    }) => str);
    const isObjectArray = objArrStr.some(o => o.isObjectArray);
    const isEcsField = fieldMaps[field] !== undefined;
    const isRuleParameters = isRuleParametersFieldOrSubfield(field, prependField);

    // Handle simple fields
    if (!isObjectArray || !isEcsField && !isRuleParameters) {
      const simpleItem = processSimpleField(dotField, strArr, isObjectArray, fieldCategory);
      resultMap.set(dotField, simpleItem);
      // eslint-disable-next-line no-continue
      continue;
    }

    // Handle threat enrichment
    if (isThreatEnrichmentFieldOrSubfield(field, prependField)) {
      const enrichmentItem = createFieldItem(fieldCategory, dotField, strArr, isObjectArray);
      resultMap.set(dotField, enrichmentItem);
    }

    // Process nested fields
    const nestedFields = processNestedFields(item, dotField, fieldCategory, isRuleParameters || isThreatEnrichmentFieldOrSubfield(field, prependField));
    // Merge results
    for (const nestedItem of nestedFields) {
      var _existing$values, _nestedItem$values;
      const existing = resultMap.get(nestedItem.field);
      if (!existing) {
        resultMap.set(nestedItem.field, nestedItem);
        // eslint-disable-next-line no-continue
        continue;
      }

      // Merge values and originalValue arrays
      const mergedValues = (_existing$values = existing.values) !== null && _existing$values !== void 0 && _existing$values.includes(((_nestedItem$values = nestedItem.values) === null || _nestedItem$values === void 0 ? void 0 : _nestedItem$values[0]) || '') ? existing.values : [...(existing.values || []), ...(nestedItem.values || [])];
      const mergedOriginal = existing.originalValue.includes(nestedItem.originalValue[0]) ? existing.originalValue : [...existing.originalValue, ...nestedItem.originalValue];
      resultMap.set(nestedItem.field, {
        ...nestedItem,
        values: mergedValues,
        originalValue: mergedOriginal
      });
    }
  }
  return Array.from(resultMap.values());
};
exports.getDataFromFieldsHits = getDataFromFieldsHits;