"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaDiscoveryService = exports.DEFAULT_TIMEOUT = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _saved_objects = require("../saved_objects");
var _config = require("../config");
var _bulk_update_error = require("../lib/bulk_update_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_TIMEOUT = exports.DEFAULT_TIMEOUT = 2000;
class KibanaDiscoveryService {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "activeNodesLookBack", void 0);
    (0, _defineProperty2.default)(this, "discoveryInterval", void 0);
    (0, _defineProperty2.default)(this, "currentNode", void 0);
    (0, _defineProperty2.default)(this, "started", false);
    (0, _defineProperty2.default)(this, "savedObjectsRepository", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "stopped", false);
    (0, _defineProperty2.default)(this, "timer", void 0);
    (0, _defineProperty2.default)(this, "onNodesCounted", void 0);
    this.activeNodesLookBack = opts.config.active_nodes_lookback;
    this.discoveryInterval = opts.config.interval;
    this.savedObjectsRepository = opts.savedObjectsRepository;
    this.logger = opts.logger;
    this.currentNode = opts.currentNode;
    this.onNodesCounted = opts.onNodesCounted;
  }
  async upsertCurrentNode({
    id,
    lastSeen
  }) {
    await this.savedObjectsRepository.update(_saved_objects.BACKGROUND_TASK_NODE_SO_NAME, id, {
      id,
      last_seen: lastSeen
    }, {
      upsert: {
        id,
        last_seen: lastSeen
      },
      refresh: false
    });
  }
  async scheduleUpsertCurrentNode() {
    let retryInterval = this.discoveryInterval;
    if (!this.stopped) {
      const lastSeenDate = new Date();
      const lastSeen = lastSeenDate.toISOString();
      try {
        await this.upsertCurrentNode({
          id: this.currentNode,
          lastSeen
        });
        if (!this.started) {
          this.logger.info('Kibana Discovery Service has been started');
          this.started = true;
        }
      } catch (e) {
        if ((0, _bulk_update_error.isClusterBlockException)(e)) {
          retryInterval = _config.DISCOVERY_INTERVAL_AFTER_BLOCK_EXCEPTION_MS;
        }
        if (!this.started) {
          this.logger.error(`Kibana Discovery Service couldn't be started and will be retried in ${retryInterval}ms, error:${e.message}`);
        } else {
          this.logger.error(`Kibana Discovery Service couldn't update this node's last_seen timestamp. id: ${this.currentNode}, last_seen: ${lastSeen}, error:${e.message}`);
        }
      } finally {
        this.timer = setTimeout(async () => await this.scheduleUpsertCurrentNode(),
        // The timeout should not be less than the default timeout of two seconds
        Math.max(retryInterval - (Date.now() - lastSeenDate.getTime()), DEFAULT_TIMEOUT));
      }
    }
  }
  isStarted() {
    return this.started;
  }
  async start() {
    if (!this.started) {
      await this.scheduleUpsertCurrentNode();
    } else {
      this.logger.warn('Kibana Discovery Service has already been started');
    }
  }
  async getActiveKibanaNodes() {
    const {
      saved_objects: activeNodes
    } = await this.savedObjectsRepository.find({
      type: _saved_objects.BACKGROUND_TASK_NODE_SO_NAME,
      perPage: 10000,
      page: 1,
      filter: `${_saved_objects.BACKGROUND_TASK_NODE_SO_NAME}.attributes.last_seen > now-${this.activeNodesLookBack}`
    });
    if (this.onNodesCounted) {
      this.onNodesCounted(activeNodes.length);
    }
    return activeNodes;
  }
  async deleteCurrentNode() {
    await this.savedObjectsRepository.delete(_saved_objects.BACKGROUND_TASK_NODE_SO_NAME, this.currentNode);
    this.logger.info('Removed this node from the Kibana Discovery Service');
  }
  stop() {
    this.stopped = true;
    if (this.timer) {
      clearTimeout(this.timer);
    }
  }
}
exports.KibanaDiscoveryService = KibanaDiscoveryService;