"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.significantEventsRoutes = void 0;
var _boom = require("@hapi/boom");
var _zod = require("@kbn/zod");
var _security_error = require("../../../lib/streams/errors/security_error");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
var _preview_significant_events = require("./preview_significant_events");
var _read_significant_events_from_alerts_indices = require("./read_significant_events_from_alerts_indices");
var _assert_enterprise_license = require("../../utils/assert_enterprise_license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Make sure strings are expected for input, but still converted to a
// Date, without breaking the OpenAPI generator
const dateFromString = _zod.z.string().transform(input => new Date(input));
const previewSignificantEventsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/significant_events/_preview 2023-10-31',
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      from: dateFromString,
      to: dateFromString,
      bucketSize: _zod.z.string()
    }),
    body: _zod.z.object({
      query: _zod.z.object({
        kql: _zod.z.object({
          query: _zod.z.string()
        })
      })
    })
  }),
  options: {
    access: 'public',
    summary: 'Preview significant events',
    description: 'Preview significant event results based on a given query',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    params,
    request,
    getScopedClients,
    logger
  }) => {
    const {
      streamsClient,
      scopedClusterClient
    } = await getScopedClients({
      request
    });
    const isStreamEnabled = await streamsClient.isStreamsEnabled();
    if (!isStreamEnabled) {
      throw (0, _boom.badRequest)('Streams is not enabled');
    }
    const {
      body: {
        query
      },
      path: {
        name
      },
      query: {
        bucketSize,
        from,
        to
      }
    } = params;
    const definition = await streamsClient.getStream(name);
    return await (0, _preview_significant_events.previewSignificantEvents)({
      definition,
      bucketSize,
      from,
      to,
      query
    }, {
      scopedClusterClient
    });
  }
});
const readSignificantEventsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/significant_events 2023-10-31',
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      from: dateFromString,
      to: dateFromString,
      bucketSize: _zod.z.string()
    })
  }),
  options: {
    access: 'public',
    summary: 'Read the significant events',
    description: 'Read the significant events',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_constants.STREAMS_TIERED_SIGNIFICANT_EVENT_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError(`Cannot access API on the current pricing tier`);
    }
    const {
      streamsClient,
      assetClient,
      scopedClusterClient,
      licensing
    } = await getScopedClients({
      request
    });
    await (0, _assert_enterprise_license.assertEnterpriseLicense)(licensing);
    const isStreamEnabled = await streamsClient.isStreamsEnabled();
    if (!isStreamEnabled) {
      throw (0, _boom.badRequest)('Streams are not enabled');
    }
    const {
      name
    } = params.path;
    const {
      from,
      to,
      bucketSize
    } = params.query;
    return await (0, _read_significant_events_from_alerts_indices.readSignificantEventsFromAlertsIndices)({
      name,
      from,
      to,
      bucketSize
    }, {
      assetClient,
      scopedClusterClient
    });
  }
});
const significantEventsRoutes = exports.significantEventsRoutes = {
  ...readSignificantEventsRoute,
  ...previewSignificantEventsRoute
};