"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.observableIntoEventSourceStream = observableIntoEventSourceStream;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _errors = require("@kbn/sse-utils/src/errors");
var _events = require("@kbn/sse-utils/src/events");
var _rxjs = require("rxjs");
var _stream = require("stream");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class ResponseStream extends _stream.PassThrough {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "_compressor", void 0);
  }
  setCompressor(compressor) {
    this._compressor = compressor;
  }
  flush() {
    var _this$_compressor;
    (_this$_compressor = this._compressor) === null || _this$_compressor === void 0 ? void 0 : _this$_compressor.flush();
  }
}
function observableIntoEventSourceStream(source$, {
  logger,
  signal,
  flushThrottleMs = 100
}) {
  const withSerializedErrors$ = source$.pipe((0, _rxjs.catchError)(error => {
    if ((0, _errors.isSSEError)(error)) {
      logger.error(error);
      logger.debug(() => JSON.stringify(error));
      return (0, _rxjs.of)({
        type: _events.ServerSentEventType.error,
        error: {
          code: error.code,
          message: error.message,
          meta: error.meta
        }
      });
    }
    logger.error(error);
    return (0, _rxjs.of)({
      type: _events.ServerSentEventType.error,
      error: {
        code: _errors.ServerSentEventErrorCode.internalError,
        message: error.message
      }
    });
  }), (0, _rxjs.map)(event => {
    const {
      type,
      ...rest
    } = event;
    return createLine({
      event: type,
      data: rest
    });
  }));
  const stream = new ResponseStream();
  const flush$ = new _rxjs.Subject();
  flush$
  // Using `leading: true` and `trailing: true` to avoid holding the flushing for too long,
  // but still avoid flushing too often (it will emit at the beginning of the throttling process, and at the end).
  .pipe((0, _rxjs.throttleTime)(flushThrottleMs, void 0, {
    leading: true,
    trailing: true
  })).subscribe(() => stream.flush());
  const intervalId = setInterval(() => {
    // `:` denotes a comment - this is to keep the connection open
    // it will be ignored by the SSE parser on the client
    stream.write(': keep-alive\n');
    flush$.next();
  }, 10000);
  const subscription = withSerializedErrors$.subscribe({
    next: line => {
      stream.write(line);
      // Make sure to flush the written lines to emit them immediately (instead of waiting for buffer to fill)
      flush$.next();
    },
    complete: () => {
      flush$.complete();
      stream.end();
      clearTimeout(intervalId);
    },
    error: error => {
      clearTimeout(intervalId);
      stream.write(createLine({
        event: 'error',
        data: {
          error: {
            code: _errors.ServerSentEventErrorCode.internalError,
            message: error.message
          }
        }
      }));
      flush$.complete();
      // No need to flush because we're ending the stream anyway
      stream.end();
    }
  });
  signal.addEventListener('abort', () => {
    subscription.unsubscribe();
    stream.end();
  });
  return stream;
}
function createLine({
  event,
  data
}) {
  return [`event: ${event}`, `data: ${JSON.stringify(data)}`,
  // We could also include `id` and `retry` if we see fit in the future.
  // https://developer.mozilla.org/en-US/docs/Web/API/Server-sent_events/Using_server-sent_events#fields
  `\n`].join(`\n`);
}