"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetSLOHealth = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LAG_THRESHOLD_MINUTES = 10;
const STALE_THRESHOLD_MINUTES = 2 * 24 * 60;
class GetSLOHealth {
  constructor(scopedClusterClient, repository) {
    this.scopedClusterClient = scopedClusterClient;
    this.repository = repository;
  }
  async execute(params) {
    const sloIds = params.list.map(({
      sloId
    }) => sloId);
    const sloList = await this.repository.findAllByIds(sloIds);
    const sloById = (0, _lodash.keyBy)(sloList, 'id');
    const filteredList = params.list.filter(item => !!sloById[item.sloId]).map(item => ({
      sloId: item.sloId,
      sloInstanceId: item.sloInstanceId,
      sloRevision: sloById[item.sloId].revision,
      sloName: sloById[item.sloId].name
    }));
    const summaryDocsById = await this.getSummaryDocsById(filteredList);
    const results = await Promise.all(filteredList.map(async item => {
      const transformStatsById = await this.getTransformStatsForSLO(item);
      const health = computeHealth(transformStatsById, item);
      const state = computeState(summaryDocsById, item);
      return {
        sloId: item.sloId,
        sloName: item.sloName,
        sloInstanceId: item.sloInstanceId,
        sloRevision: item.sloRevision,
        state,
        health
      };
    }));

    /*
     * Map results based on SLO ids since transforms represent all instances
     * Since "state" is not being used in Kibana, we can group by SLO id and return only one result per SLO
     * If needed in the future, we can return all instances by removing this mapping
     * and adding sloInstanceId to the response schema
     */
    const mappedResults = Array.from(new Map(results.map(item => [`${item.sloId}-${item.sloRevision}`, item])).values());
    return _sloSchema.fetchSLOHealthResponseSchema.encode(mappedResults);
  }
  async getSummaryDocsById(filteredList) {
    const summaryDocs = await this.scopedClusterClient.asCurrentUser.search({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      query: {
        bool: {
          should: filteredList.map(item => ({
            bool: {
              must: [{
                term: {
                  'slo.id': item.sloId
                }
              }, {
                term: {
                  'slo.instanceId': item.sloInstanceId
                }
              }]
            }
          }))
        }
      }
    });
    const summaryDocsById = (0, _lodash.groupBy)(summaryDocs.hits.hits.map(hit => hit._source), doc => buildSummaryKey(doc.slo.id, doc.slo.instanceId));
    return summaryDocsById;
  }
  async getTransformStatsForSLO(item) {
    const rollupTransformStats = await this.scopedClusterClient.asSecondaryAuthUser.transform.getTransformStats({
      transform_id: (0, _constants.getSLOTransformId)(item.sloId, item.sloRevision),
      allow_no_match: true,
      size: 1
    }, {
      ignore: [404]
    });
    const summaryTransformStats = await this.scopedClusterClient.asSecondaryAuthUser.transform.getTransformStats({
      transform_id: (0, _constants.getSLOSummaryTransformId)(item.sloId, item.sloRevision),
      allow_no_match: true,
      size: 1
    }, {
      ignore: [404]
    });
    const allTransforms = [...(rollupTransformStats.transforms || []), ...(summaryTransformStats.transforms || [])];
    return (0, _lodash.keyBy)(allTransforms, transform => transform.id);
  }
}
exports.GetSLOHealth = GetSLOHealth;
function buildSummaryKey(id, instanceId) {
  return id + '|' + instanceId;
}
function computeState(summaryDocsById, item) {
  const sloSummaryDocs = summaryDocsById[buildSummaryKey(item.sloId, item.sloInstanceId)];
  let state = 'no_data';
  if (!sloSummaryDocs) {
    return state;
  }
  const hasOnlyTempSummaryDoc = sloSummaryDocs.every(doc => doc.isTempDoc); // only temporary documents mean the summary transform did not run yet
  const sloSummarydoc = sloSummaryDocs.find(doc => !doc.isTempDoc);
  const latestSliTimestamp = sloSummarydoc === null || sloSummarydoc === void 0 ? void 0 : sloSummarydoc.latestSliTimestamp;
  const summaryUpdatedAt = sloSummarydoc === null || sloSummarydoc === void 0 ? void 0 : sloSummarydoc.summaryUpdatedAt;
  if (hasOnlyTempSummaryDoc) {
    state = 'no_data';
  } else if (summaryUpdatedAt && latestSliTimestamp) {
    const summaryLag = (0, _moment.default)().diff(new Date(summaryUpdatedAt), 'minute');
    const indexingLag = (0, _moment.default)(summaryUpdatedAt).diff(new Date(latestSliTimestamp), 'minute');

    // When the summaryUpdatedAt is greater than STALE_THRESHOLD_MINUTES minutes, the SLO is considered stale since no new data triggered a summary document update.
    // When the difference between the summaryUpdatedAt and the latestSliTimestamp is
    // - Below LAG_THRESHOLD_MINUTES minutes, the SLO has cought up with the sli data, and is running correctly
    // - Above LAG_THRESHOLD_MINUTES minutes, the SLO is indexing
    if (summaryLag > STALE_THRESHOLD_MINUTES) {
      state = 'stale';
    } else {
      state = indexingLag >= LAG_THRESHOLD_MINUTES ? 'indexing' : 'running';
    }
  }
  return state;
}
function getTransformHealth(transformStat) {
  var _transformStat$health, _transformStat$health2;
  if (!transformStat) {
    return 'missing';
  }
  return ((_transformStat$health = transformStat.health) === null || _transformStat$health === void 0 ? void 0 : (_transformStat$health2 = _transformStat$health.status) === null || _transformStat$health2 === void 0 ? void 0 : _transformStat$health2.toLowerCase()) === 'green' ? 'healthy' : 'unhealthy';
}
function computeHealth(transformStatsById, item) {
  const rollup = getTransformHealth(transformStatsById[(0, _constants.getSLOTransformId)(item.sloId, item.sloRevision)]);
  const summary = getTransformHealth(transformStatsById[(0, _constants.getSLOSummaryTransformId)(item.sloId, item.sloRevision)]);
  const overall = rollup === 'healthy' && summary === 'healthy' ? 'healthy' : 'unhealthy';
  return {
    overall,
    rollup,
    summary
  };
}